<?php
// Disable error display to prevent HTML output
error_reporting(0);
ini_set('display_errors', 0);

// Try to increase upload limits (may not work on all servers)
ini_set('upload_max_filesize', '50M');
ini_set('post_max_size', '50M');
ini_set('max_execution_time', 300);
ini_set('max_input_time', 300);
ini_set('memory_limit', '256M');

session_start();
require_once '../includes/auth.php';
require_once '../includes/functions.php';

header('Content-Type: application/json');

try {
    // Check if admin is logged in
    $adminAuth = new AdminAuth();
    if (!$adminAuth->isLoggedIn()) {
        echo json_encode(['success' => false, 'message' => 'Unauthorized']);
        exit();
    }

    if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
        echo json_encode(['success' => false, 'message' => 'Invalid request method']);
        exit();
    }

    if (!isset($_FILES['banner_image'])) {
        echo json_encode(['success' => false, 'message' => 'No file uploaded']);
        exit();
    }

    $file = $_FILES['banner_image'];

    if ($file['error'] !== UPLOAD_ERR_OK) {
        $errorMessages = [
            UPLOAD_ERR_INI_SIZE => 'File too large. Server limit: ' . ini_get('upload_max_filesize') . '. Try a smaller file.',
            UPLOAD_ERR_FORM_SIZE => 'File too large. Form limit exceeded.',
            UPLOAD_ERR_PARTIAL => 'File upload was interrupted.',
            UPLOAD_ERR_NO_FILE => 'No file was uploaded.',
            UPLOAD_ERR_NO_TMP_DIR => 'Missing temporary folder.',
            UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
            UPLOAD_ERR_EXTENSION => 'File upload stopped by extension.'
        ];
        
        $errorMessage = isset($errorMessages[$file['error']]) ? $errorMessages[$file['error']] : 'Unknown upload error: ' . $file['error'];
        echo json_encode(['success' => false, 'message' => $errorMessage]);
        exit();
    }

    $allowedTypes = [
        'image/jpeg', 'image/jpg', 'image/png', 'image/gif',
        'video/mp4', 'video/webm', 'video/ogg', 'video/avi', 'video/quicktime'
    ];

    $maxSize = 50 * 1024 * 1024; // 50MB (increased back)

    // Check file type using $_FILES['type'] instead of finfo
    if (!in_array($file['type'], $allowedTypes)) {
        echo json_encode(['success' => false, 'message' => 'Invalid file type. Allowed: JPG, PNG, GIF, MP4, WebM, OGG, AVI, MOV']);
        exit();
    }

    if ($file['size'] > $maxSize) {
        echo json_encode(['success' => false, 'message' => 'File size too large. Maximum size is 50MB.']);
        exit();
    }

    // Create upload directory if it doesn't exist
    $uploadDir = __DIR__ . '/../../banner/';
    if (!is_dir($uploadDir)) {
        if (!mkdir($uploadDir, 0755, true)) {
            echo json_encode(['success' => false, 'message' => 'Failed to create banner directory']);
            exit();
        }
    }

    // Get file extension from original filename
    $fileExtension = strtolower(pathinfo($file['name'], PATHINFO_EXTENSION));
    $allowedExtensions = ['jpg', 'jpeg', 'png', 'gif', 'mp4', 'webm', 'ogg', 'avi', 'mov'];
    
    if (!in_array($fileExtension, $allowedExtensions)) {
        echo json_encode(['success' => false, 'message' => 'Invalid file extension']);
        exit();
    }

    $filename = 'banner_' . time() . '_' . uniqid() . '.' . $fileExtension;
    $targetPath = $uploadDir . $filename;

    if (!move_uploaded_file($file['tmp_name'], $targetPath)) {
        echo json_encode(['success' => false, 'message' => 'Failed to save file']);
        exit();
    }

    // Build public path
    $publicPath = 'banner/' . $filename;
    $isVideo = in_array($fileExtension, ['mp4', 'webm', 'ogg', 'avi', 'mov']);

    echo json_encode([
        'success' => true, 
        'message' => 'Banner uploaded successfully!',
        'filename' => $filename,
        'path' => $publicPath,
        'type' => $isVideo ? 'video' : 'image'
    ]);

} catch (Exception $e) {
    echo json_encode(['success' => false, 'message' => 'Server error: ' . $e->getMessage()]);
}
?>