<?php
require_once '../includes/auth.php';
require_once '../includes/functions.php';

$adminAuth->requireLogin();

// Get delivered orders
$deliveredOrders = $adminFunctions->getDeliveredOrders();

// Get delivered orders statistics
$deliveredStats = $adminFunctions->getDeliveredStats();
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Delivered Orders - Admin Panel</title>
    <link rel="stylesheet" href="../assets/css/admin.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
</head>
<body>
    <!-- Sidebar -->
    <div class="sidebar">
        <div class="sidebar-header">
            <img src="../../images/LOGO.png" alt="Logo" class="sidebar-logo">
            <h3>Admin Panel</h3>
        </div>
        
        <nav class="sidebar-nav">
            <a href="../index.php" class="nav-item">
                <i class="fas fa-th-large"></i>
                <span>Overview</span>
            </a>
            <a href="products.php" class="nav-item">
                <i class="fas fa-box"></i>
                <span>Products</span>
            </a>
            <a href="orders.php" class="nav-item">
                <i class="fas fa-shopping-cart"></i>
                <span>Orders</span>
            </a>
            <a href="delivered.php" class="nav-item active">
                <i class="fas fa-check-circle"></i>
                <span>Delivered</span>
            </a>
            <a href="customers.php" class="nav-item">
                <i class="fas fa-users"></i>
                <span>Customers</span>
            </a>
            <a href="feedback.php" class="nav-item">
                <i class="fas fa-comments"></i>
                <span>Feedback</span>
            </a>
            <a href="shipping.php" class="nav-item">
                <i class="fas fa-shipping-fast"></i>
                <span>Shipping</span>
            </a>
            <a href="banners.php" class="nav-item">
                <i class="fas fa-image"></i>
                <span>Banners</span>
            </a>
            <a href="presets.php" class="nav-item">
                <i class="fas fa-cogs"></i>
                <span>Field Presets</span>
            </a>
            <a href="vouchers.php" class="nav-item">
                <i class="fas fa-ticket-alt"></i>
                <span>Voucher Management</span>
            </a>
            <a href="notifications.php" class="nav-item">
                <i class="fas fa-bell"></i>
                <span>Push Notifications</span>
            </a>
            <a href="analytics.php" class="nav-item">
                <i class="fas fa-chart-bar"></i>
                <span>Analytics</span>
            </a>
            <a href="settings.php" class="nav-item">
                <i class="fas fa-cog"></i>
                <span>Settings</span>
            </a>
        </nav>
        
        <div class="sidebar-footer">
            <div class="user-info">
                <i class="fas fa-user-circle"></i>
                <span><?php echo htmlspecialchars($adminAuth->getAdminName()); ?></span>
            </div>
            <a href="../logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Logout</span>
            </a>
        </div>
    </div>

    <!-- Main Content -->
    <div class="main-content">
        <!-- Header -->
        <header class="header">
            <div class="header-left">
                <button class="sidebar-toggle">
                    <i class="fas fa-bars"></i>
                </button>
                <h1>Delivered Orders</h1>
            </div>
            <div class="header-right">
                <div class="header-actions">
                    <button onclick="exportDeliveredOrders()" class="btn btn-outline">
                        <i class="fas fa-download"></i>
                        Export Delivered
                    </button>
                    <a href="../index.php" target="_blank" class="btn btn-outline">
                        <i class="fas fa-external-link-alt"></i>
                        View Store
                    </a>
                </div>
            </div>
        </header>

        <!-- Page Content -->
        <div class="page-content">
            <!-- Statistics Cards -->
            <div class="stats-grid">
                <div class="stat-card delivered">
                    <div class="stat-icon">
                        <i class="fas fa-check-circle"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($deliveredStats['total_delivered']); ?></h3>
                        <p>Total Delivered</p>
                    </div>
                </div>
                
                <div class="stat-card revenue">
                    <div class="stat-icon">
                        <i class="fas fa-dollar-sign"></i>
                    </div>
                    <div class="stat-content">
                        <h3>RM <?php echo number_format($deliveredStats['total_revenue'], 2); ?></h3>
                        <p>Delivered Revenue</p>
                    </div>
                </div>
                
                <div class="stat-card this-month">
                    <div class="stat-icon">
                        <i class="fas fa-calendar-check"></i>
                    </div>
                    <div class="stat-content">
                        <h3><?php echo number_format($deliveredStats['this_month']); ?></h3>
                        <p>This Month</p>
                    </div>
                </div>
                
                <div class="stat-card avg-value">
                    <div class="stat-icon">
                        <i class="fas fa-chart-line"></i>
                    </div>
                    <div class="stat-content">
                        <h3>RM <?php echo number_format($deliveredStats['avg_order_value'], 2); ?></h3>
                        <p>Avg Order Value</p>
                    </div>
                </div>
            </div>

            <!-- Search and Filter -->
            <div class="search-filter">
                <input type="text" class="search-input" placeholder="Search delivered orders..." id="orderSearch">
                <select class="filter-select" id="dateRangeFilter">
                    <option value="">All Time</option>
                    <option value="today">Today</option>
                    <option value="week">This Week</option>
                    <option value="month">This Month</option>
                    <option value="quarter">This Quarter</option>
                    <option value="year">This Year</option>
                </select>
                <input type="date" class="filter-select" id="dateFilter" placeholder="Filter by date">
            </div>

            <!-- Delivered Orders Table -->
            <div class="table-container">
                <div class="table-content">
                    <table class="data-table" id="deliveredTable">
                        <thead>
                            <tr>
                                <th>Order ID</th>
                                <th>Customer</th>
                                <th>Email</th>
                                <th>Total</th>
                                <th>Payment</th>
                                <th>Delivered Date</th>
                                <th>Delivery Time</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (empty($deliveredOrders)): ?>
                            <tr>
                                <td colspan="8" class="no-data">
                                    <div class="no-data-content">
                                        <i class="fas fa-check-circle"></i>
                                        <h3>No Delivered Orders</h3>
                                        <p>There are no delivered orders to display at the moment.</p>
                                    </div>
                                </td>
                            </tr>
                            <?php else: ?>
                            <?php foreach ($deliveredOrders as $order): ?>
                            <tr>
                                <td>
                                    <strong><?php echo $order['order_number']; ?></strong>
                                </td>
                                <td>
                                    <?php 
                                    $shipping_address = json_decode($order['shipping_address'], true);
                                    if ($shipping_address) {
                                        echo htmlspecialchars($shipping_address['first_name'] . ' ' . $shipping_address['last_name']);
                                    } else {
                                        echo 'N/A';
                                    }
                                    ?>
                                </td>
                                <td>
                                    <?php 
                                    if ($shipping_address) {
                                        echo htmlspecialchars($shipping_address['email']);
                                    } else {
                                        echo 'N/A';
                                    }
                                    ?>
                                </td>
                                <td>RM <?php echo number_format($order['total_amount'], 2); ?></td>
                                <td>
                                    <span class="payment-method">
                                        <i class="fas fa-credit-card"></i>
                                        <?php echo htmlspecialchars($order['payment_method']); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="delivery-date">
                                        <i class="fas fa-calendar-check"></i>
                                        <?php echo date('M j, Y', strtotime($order['updated_at'])); ?>
                                    </span>
                                </td>
                                <td>
                                    <span class="delivery-time">
                                        <i class="fas fa-clock"></i>
                                        <?php 
                                        $deliveryTime = strtotime($order['updated_at']) - strtotime($order['created_at']);
                                        $days = floor($deliveryTime / (24 * 60 * 60));
                                        $hours = floor(($deliveryTime % (24 * 60 * 60)) / (60 * 60));
                                        
                                        if ($days > 0) {
                                            echo $days . ' day' . ($days > 1 ? 's' : '') . ' ' . $hours . ' hour' . ($hours > 1 ? 's' : '');
                                        } else {
                                            echo $hours . ' hour' . ($hours > 1 ? 's' : '');
                                        }
                                        ?>
                                    </span>
                                </td>
                                <td>
                                    <div class="action-buttons">
                                        <button onclick="viewOrder(<?php echo $order['id']; ?>)" 
                                                class="btn btn-sm btn-primary" title="View Details">
                                            <i class="fas fa-eye"></i>
                                        </button>
                                        <button onclick="printInvoice(<?php echo $order['id']; ?>)" 
                                                class="btn btn-sm btn-success" title="Print Invoice">
                                            <i class="fas fa-print"></i>
                                        </button>
                                        <button onclick="resendConfirmation(<?php echo $order['id']; ?>)" 
                                                class="btn btn-sm btn-info" title="Resend Confirmation">
                                            <i class="fas fa-envelope"></i>
                                        </button>
                                    </div>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </div>
        </div>
    </div>

    <!-- Order Details Modal -->
    <div id="orderDetailsModal" class="modal">
        <div class="modal-content" style="max-width: 800px;">
            <div class="modal-header">
                <h3 class="modal-title">Delivered Order Details</h3>
                <span class="close" onclick="hideModal('orderDetailsModal')">&times;</span>
            </div>
            
            <div id="orderDetailsContent">
                <!-- Order details will be loaded here -->
            </div>
        </div>
    </div>

    <script src="../assets/js/admin.js"></script>
    <script>
        function viewOrder(orderId) {
            // Load order details via AJAX
            fetch(`ajax/get_order_details.php?id=${orderId}`)
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        document.getElementById('orderDetailsContent').innerHTML = data.html;
                        showModal('orderDetailsModal');
                    } else {
                        alert('Error loading order details');
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error loading order details');
                });
        }
        
        function printInvoice(orderId) {
            // Open print window for invoice
            window.open(`ajax/print_invoice.php?id=${orderId}`, '_blank', 'width=800,height=600');
        }
        
        function resendConfirmation(orderId) {
            if (confirm('Are you sure you want to resend the delivery confirmation email?')) {
                fetch('ajax/resend_confirmation.php', {
                    method: 'POST',
                    headers: {
                        'Content-Type': 'application/x-www-form-urlencoded',
                    },
                    body: `order_id=${orderId}`
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        alert('Confirmation email sent successfully!');
                    } else {
                        alert('Error sending confirmation email: ' + data.message);
                    }
                })
                .catch(error => {
                    console.error('Error:', error);
                    alert('Error sending confirmation email');
                });
            }
        }
        
        function exportDeliveredOrders() {
            const table = document.getElementById('deliveredTable');
            const rows = Array.from(table.querySelectorAll('tr'));
            const csvContent = rows.map(row => 
                Array.from(row.querySelectorAll('th, td')).map(cell => 
                    `"${cell.textContent.trim()}"`
                ).join(',')
            ).join('\n');
            
            const blob = new Blob([csvContent], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'delivered_orders_export.csv';
            a.click();
            window.URL.revokeObjectURL(url);
        }
        
        // Search functionality
        document.getElementById('orderSearch').addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#deliveredTable tbody tr');
            
            rows.forEach(row => {
                const text = row.textContent.toLowerCase();
                if (text.includes(searchTerm)) {
                    row.style.display = '';
                } else {
                    row.style.display = 'none';
                }
            });
        });
        
        // Date range filter
        document.getElementById('dateRangeFilter').addEventListener('change', function() {
            const filterValue = this.value;
            const rows = document.querySelectorAll('#deliveredTable tbody tr');
            const now = new Date();
            
            rows.forEach(row => {
                const dateCell = row.cells[5]; // Delivered date column
                if (dateCell && dateCell.textContent.trim() !== '') {
                    const cellDate = new Date(dateCell.textContent.trim());
                    let showRow = true;
                    
                    switch(filterValue) {
                        case 'today':
                            showRow = cellDate.toDateString() === now.toDateString();
                            break;
                        case 'week':
                            const weekAgo = new Date(now.getTime() - 7 * 24 * 60 * 60 * 1000);
                            showRow = cellDate >= weekAgo;
                            break;
                        case 'month':
                            const monthAgo = new Date(now.getTime() - 30 * 24 * 60 * 60 * 1000);
                            showRow = cellDate >= monthAgo;
                            break;
                        case 'quarter':
                            const quarterAgo = new Date(now.getTime() - 90 * 24 * 60 * 60 * 1000);
                            showRow = cellDate >= quarterAgo;
                            break;
                        case 'year':
                            const yearAgo = new Date(now.getTime() - 365 * 24 * 60 * 60 * 1000);
                            showRow = cellDate >= yearAgo;
                            break;
                    }
                    
                    row.style.display = showRow ? '' : 'none';
                }
            });
        });
        
        // Date filter
        document.getElementById('dateFilter').addEventListener('change', function() {
            const filterDate = this.value;
            const rows = document.querySelectorAll('#deliveredTable tbody tr');
            
            rows.forEach(row => {
                const dateCell = row.cells[5]; // Delivered date column
                if (dateCell && dateCell.textContent.trim() !== '') {
                    const cellDate = new Date(dateCell.textContent.trim());
                    const filterDateObj = new Date(filterDate);
                    
                    if (filterDate === '' || 
                        cellDate.toDateString() === filterDateObj.toDateString()) {
                        row.style.display = '';
                    } else {
                        row.style.display = 'none';
                    }
                }
            });
        });
    </script>
</body>
</html>