<?php
session_start();
require_once 'models/Cart.php';
require_once 'models/Product.php';
require_once 'models/Loyalty.php';

// Get cart items from database
$cart = new Cart();
$product = new Product();
$loyalty = new Loyalty();
$user_id = $_SESSION['user_id'] ?? null;
$session_id = session_id();

$cart_items = $cart->getCartItems($user_id, $session_id);

// Get current product pricing for each cart item
$updated_cart_items = [];
foreach ($cart_items as $item) {
    // Get current product data with pricing and variations
    $current_product = $product->getById($item['product_id']);
    if ($current_product) {
        // Find the specific variation that matches the cart item's size AND version
        $matching_variation = null;
        if (!empty($current_product['variations_detail']) && !empty($item['size'])) {
            foreach ($current_product['variations_detail'] as $variation) {
                // Match both size and version if version is specified
                if ($variation['size'] === $item['size']) {
                    if (!empty($item['version']) && !empty($variation['version_type'])) {
                        // Both have version info, match both
                        if ($variation['version_type'] === $item['version']) {
                            $matching_variation = $variation;
                            break;
                        }
                    } elseif (empty($item['version']) && empty($variation['version_type'])) {
                        // Neither has version info, match by size only
                        $matching_variation = $variation;
                        break;
                    } elseif (empty($item['version']) && !empty($variation['version_type'])) {
                        // Cart item has no version but variation does, skip this variation
                        continue;
                    } elseif (!empty($item['version']) && empty($variation['version_type'])) {
                        // Cart item has version but variation doesn't, skip this variation
                        continue;
                    }
                }
            }
        }
        
        // Update the cart item with current pricing
        $item['current_price'] = $item['price']; // Keep original price for reference
        $item['variations_detail'] = $current_product['variations_detail'] ?? [];
        $item['flash_sale_variations_detail'] = $current_product['flash_sale_variations_detail'] ?? [];
        
        // Use variation-specific pricing if available
        if ($matching_variation) {
            $item['display_price'] = $matching_variation['price'];
            $item['variation_stock'] = $matching_variation['stock'];
            $item['original_price'] = $matching_variation['price'];
            $item['sale_price'] = $matching_variation['price'];
            $item['is_on_sale'] = false;
            
            // Check if this specific variation is on flash sale
            if (!empty($item['flash_sale_variations_detail'])) {
                foreach ($item['flash_sale_variations_detail'] as $saleVariation) {
                    if ($saleVariation['version_type'] === $matching_variation['version_type'] && 
                        $saleVariation['size'] === $matching_variation['size']) {
                        $item['sale_price'] = $saleVariation['sale_price'];
                        $item['original_price'] = $saleVariation['original_price'];
                        $item['is_on_sale'] = true;
                        $item['effective_price'] = $saleVariation['sale_price'];
                        break;
                    }
                }
            }
            
            // Check for regular sale if not on flash sale
            if (!$item['is_on_sale']) {
                $item['sale_active'] = $current_product['sale_active'] ?? false;
                $item['flash_sale_active'] = $current_product['flash_sale_active'] ?? false;
                
                if ($item['sale_active'] && !empty($current_product['sale_price'])) {
                    $item['sale_price'] = $current_product['sale_price'];
                    $item['is_on_sale'] = true;
                    $item['effective_price'] = $current_product['sale_price'];
                } else {
                    $item['effective_price'] = $matching_variation['price'];
                }
            }
        } else {
            // Fallback to product-level pricing
            $item['display_price'] = $current_product['display_price'] ?? $item['price'];
            $item['original_price'] = $current_product['display_price'] ?? $item['price'];
            $item['sale_price'] = $current_product['sale_price'] ?? null;
            $item['sale_active'] = $current_product['sale_active'] ?? false;
            $item['flash_sale_active'] = $current_product['flash_sale_active'] ?? false;
            $item['is_on_sale'] = false;
            
            if ($item['sale_active'] && !empty($item['sale_price'])) {
                $item['effective_price'] = $item['sale_price'];
            } else {
                $item['effective_price'] = $item['display_price'];
            }
        }
        
        $item['is_out_of_stock'] = $current_product['is_out_of_stock'] ?? false;
    } else {
        $item['effective_price'] = $item['price'];
    }
    
    $updated_cart_items[] = $item;
}

$cart_items = $updated_cart_items;

// Recalculate totals with effective prices
$subtotal = 0;
$item_count = count($cart_items);
$total_quantity = 0;

foreach ($cart_items as $item) {
    $effective_price = $item['effective_price'] ?? $item['price'];
    $subtotal += $effective_price * $item['quantity'];
    $total_quantity += $item['quantity'];
}

// Calculate shipping
$weight = 0.1; // Fixed weight for perfume orders (perfume bottles are very light)
$shipping_cost = $cart->calculateShippingCost($subtotal, $weight);

// Calculate total points from cart items based on price (RM1 = 1 point)
$total_points = 0;
foreach ($cart_items as $item) {
    // Calculate points based on price instead of manual point_rewards
    $item_price = $item['price'] ?? 0;
    if ($item_price > 0) {
        $points_per_item = $loyalty->calculatePointsFromPrice($item_price);
        $total_points += $points_per_item * $item['quantity'];
    }
}

$cart_total = [
    'subtotal' => $subtotal,
    'shipping' => $shipping_cost,
    'total' => $subtotal + $shipping_cost,
    'item_count' => $item_count,
    'total_quantity' => $total_quantity,
    'total_points' => $total_points
];

// Get shipping information
$shipping_info = null;
if (!empty($cart_items) && $cart_total['subtotal'] > 0) {
    $shipping_info = $cart->getShippingInfo($cart_total['subtotal'], $weight);
}

// Get current shipping configuration for JavaScript
$shipping_config = null;
// Always get shipping config, even for empty cart (for when items are added)
$order_amount = $cart_total['subtotal'] ?? 0;
$shipping_config = $cart->getShippingConfig($order_amount);
// Debug: Log the order amount being used
error_log("Cart page requesting shipping config for order amount: " . $order_amount);

// Debug information (remove in production)
echo "<!-- Debug: user_id = " . ($user_id ?? 'null') . ", session_id = " . $session_id . " -->";
echo "<!-- Debug: cart_items count = " . count($cart_items) . " -->";
echo "<!-- Debug: subtotal = " . $subtotal . ", shipping_cost = " . $shipping_cost . ", total = " . ($subtotal + $shipping_cost) . " -->";
echo "<!-- Debug: shipping_config = " . json_encode($shipping_config) . " -->";
foreach ($cart_items as $i => $item) {
    echo "<!-- Debug Item $i: name=" . htmlspecialchars($item['name']) . ", size=" . htmlspecialchars($item['size'] ?? 'null') . ", price=" . $item['price'] . ", effective_price=" . ($item['effective_price'] ?? 'null') . " -->";
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cart - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="cart.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <?php include_once 'includes/appilix-sdk.php'; ?>
    <?php include_once 'includes/appilix-user-identity.php'; ?>
    <script src="script.js" defer></script>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="goBack()"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <div class="cart-icon">
                <i class="fas fa-shopping-cart"></i>
                <span class="cart-count" style="<?php echo count($cart_items) > 0 ? 'display: inline-block;' : 'display: none;'; ?>"><?php echo count($cart_items); ?></span>
            </div>
        </div>
    </nav>

    <!-- Cart Header -->
    <div class="cart-header">
        <h1>My Cart</h1>
        <p class="cart-subtitle">Review your selected items</p>
    </div>

    <!-- Cart Items -->
    <main class="cart-main">
        <div id="cart-items-container">
            <?php if (!empty($cart_items)): ?>
                <!-- Select All Section -->
                <div class="select-all-section">
                    <div class="select-all-checkbox">
                        <input type="checkbox" id="select-all-items" checked onchange="toggleSelectAll()">
                        <label for="select-all-items" class="checkbox-label">
                            <span class="select-all-text">Select All Items</span>
                        </label>
                    </div>
                    <div class="selected-count">
                        <span id="selected-count"><?php echo count($cart_items); ?></span> of <?php echo count($cart_items); ?> selected
                    </div>
                </div>
                <?php foreach ($cart_items as $item): ?>
                <div class="cart-item" data-cart-id="<?php echo $item['id']; ?>">
                    <div class="cart-item-checkbox">
                        <input type="checkbox" class="item-select-checkbox" 
                               id="item_<?php echo $item['id']; ?>" 
                               data-cart-id="<?php echo $item['id']; ?>"
                               data-price="<?php echo $item['effective_price'] ?? $item['price']; ?>"
                               data-quantity="<?php echo $item['quantity']; ?>"
                               checked
                               onchange="updateSelectedItems()">
                    </div>
                    <div class="cart-item-image">
                        <img src="<?php echo htmlspecialchars($item['image']); ?>" alt="<?php echo htmlspecialchars($item['name']); ?>" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjUwIiB5PSI1MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOTk5IiB0ZXh0LWFuY2hvcj0ibWlkZGxlIj5JbWFnZTwvdGV4dD48L3N2Zz4='">
                        
                        <!-- Flash Sale Badge -->
                        <?php if (isset($item['flash_sale_active']) && $item['flash_sale_active'] && isset($item['is_on_sale']) && $item['is_on_sale']): ?>
                            <div class="cart-flash-sale-badge">
                                <i class="fas fa-bolt"></i>
                                <span>FLASH SALE</span>
                            </div>
                        <?php elseif (isset($item['sale_active']) && $item['sale_active'] && isset($item['is_on_sale']) && $item['is_on_sale']): ?>
                            <div class="cart-sale-badge">
                                <i class="fas fa-tag"></i>
                                <span>SALE</span>
                            </div>
                        <?php endif; ?>
                    </div>
                    <div class="cart-item-details">
                        <h3 class="cart-item-name"><?php echo htmlspecialchars($item['name']); ?></h3>
                        <?php if ($item['size']): ?>
                            <p class="cart-item-size">
                                <?php if (!empty($item['version'])): ?>
                                    Version: <?php echo htmlspecialchars($item['version']); ?> | 
                                <?php endif; ?>
                                Size: <?php echo htmlspecialchars($item['size']); ?>
                            </p>
                        <?php endif; ?>
                        
                        <?php if (isset($item['is_out_of_stock']) && $item['is_out_of_stock']): ?>
                            <div class="out-of-stock-warning">
                                <i class="fas fa-exclamation-triangle"></i>
                                <span>This item is currently out of stock</span>
                            </div>
                        <?php elseif (isset($item['variation_stock']) && $item['variation_stock'] < $item['quantity']): ?>
                            <div class="low-stock-warning">
                                <i class="fas fa-exclamation-circle"></i>
                                <span>Only <?php echo $item['variation_stock']; ?> left in stock</span>
                            </div>
                        <?php endif; ?>
                        <div class="cart-item-price-container">
                            <?php if (isset($item['is_on_sale']) && $item['is_on_sale'] && isset($item['sale_price']) && isset($item['original_price'])): ?>
                                <div class="cart-price-container">
                                    <span class="cart-sale-price">RM <?php echo number_format($item['sale_price'], 2); ?></span>
                                    <span class="cart-original-price">RM <?php echo number_format($item['original_price'], 2); ?></span>
                                </div>
                            <?php else: ?>
                                <p class="cart-item-price">RM <?php echo number_format($item['effective_price'] ?? $item['price'], 2); ?></p>
                            <?php endif; ?>
                        </div>
                    </div>
                    <div class="cart-item-controls">
                        <div class="quantity-controls">
                            <button class="quantity-btn minus-btn" data-cart-id="<?php echo $item['id']; ?>">
                                <i class="fas fa-minus"></i>
                            </button>
                            <span class="quantity"><?php echo $item['quantity']; ?></span>
                            <button class="quantity-btn plus-btn" data-cart-id="<?php echo $item['id']; ?>">
                                <i class="fas fa-plus"></i>
                            </button>
                        </div>
                        <button class="remove-btn" onclick="removeFromCart(<?php echo $item['id']; ?>)">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </div>
                <?php endforeach; ?>
            <?php else: ?>
                <div id="empty-cart" class="empty-cart">
                    <div class="empty-cart-icon">
                        <i class="fas fa-shopping-cart"></i>
                    </div>
                    <h3>Your cart is empty</h3>
                    <p>Add some products to get started!</p>
                    <button class="continue-shopping-btn" onclick="window.location.href='products.php'">
                        Continue Shopping
                    </button>
                </div>
            <?php endif; ?>
        </div>
    </main>

    <!-- Cart Summary -->
    <?php if (!empty($cart_items)): ?>
    <div class="cart-summary" id="cart-summary">
        <div class="summary-row">
            <span>Subtotal</span>
            <span id="subtotal">RM <?php echo number_format($cart_total['subtotal'] ?? 0, 2); ?></span>
        </div>
        <div class="summary-row">
            <div class="shipping-info">
                <span>Shipping</span>
                <small class="shipping-method" id="shipping-method">
                    <?php if ($shipping_info): ?>
                        <?php echo htmlspecialchars($shipping_info['name']); ?>
                    <?php else: ?>
                        Standard Shipping
                    <?php endif; ?>
                </small>
                <small class="delivery-time" id="delivery-time">
                <?php if ($shipping_info): ?>
                    <?php if ($shipping_info['delivery_days_min'] == $shipping_info['delivery_days_max']): ?>
                            <?php echo $shipping_info['delivery_days_min']; ?> day<?php echo $shipping_info['delivery_days_min'] != 1 ? 's' : ''; ?>
                        <?php else: ?>
                            <?php echo $shipping_info['delivery_days_min']; ?>-<?php echo $shipping_info['delivery_days_max']; ?> days
                        <?php endif; ?>
                    <?php else: ?>
                        3-5 days
                    <?php endif; ?>
                </small>
            </div>
            <span id="shipping-cost">
                <?php if ($cart_total['shipping'] > 0): ?>
                    RM <?php echo number_format($cart_total['shipping'], 2); ?>
                <?php else: ?>
                    Free
                <?php endif; ?>
            </span>
        </div>
        <?php if ($cart_total['total_points'] > 0): ?>
        <div class="summary-row points-rewards">
            <div class="points-info">
                <span>Loyalty Points</span>
                <small class="points-description">You'll earn these points after purchase</small>
            </div>
            <span class="points-value"><?php echo number_format($cart_total['total_points']); ?> pts</span>
        </div>
        <?php endif; ?>
        <div class="summary-row total">
            <span>Total</span>
            <span id="total">RM <?php echo number_format($cart_total['total'] ?? 0, 2); ?></span>
        </div>
        <button class="checkout-btn" id="checkout-btn" onclick="proceedToCheckout()">
            Proceed to Checkout
        </button>
    </div>
    <?php endif; ?>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='products.php'">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item active">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item" onclick="checkRegisteredUserAndRedirect('royalty-rewards.php')">
            <i class="fas fa-crown"></i>
        </div>
    </nav>

    <!-- Loading Indicator -->
    <div id="cart-loading" class="cart-loading">
        <div class="cart-loading-spinner"></div>
        <span>Calculating...</span>
    </div>

    <script>
        // Shipping configuration from server
        const shippingConfig = <?php echo json_encode($shipping_config); ?>;
        
        // Debug shipping config
        console.log('Shipping config loaded:', shippingConfig);
        console.log('Shipping config type:', shippingConfig ? shippingConfig.fee_type : 'null');
        console.log('Shipping config base_fee:', shippingConfig ? shippingConfig.base_fee : 'null');
        console.log('Shipping config per_kg_fee:', shippingConfig ? shippingConfig.per_kg_fee : 'null');
        
        function goBack() {
            window.history.back();
        }


        // Loading functions
        function showLoading() {
            const loading = document.getElementById('cart-loading');
            if (loading) {
                loading.classList.add('show');
            }
        }

        function hideLoading() {
            const loading = document.getElementById('cart-loading');
            if (loading) {
                loading.classList.remove('show');
            }
        }

        // Show notification function
        function showNotification(message, type = 'success') {
            const notification = document.createElement('div');
            notification.className = `notification ${type}`;
            notification.textContent = message;
            
            // Style the notification
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                right: 20px;
                padding: 15px 20px;
                border-radius: 8px;
                color: white;
                font-weight: 600;
                z-index: 9999;
                max-width: 300px;
                box-shadow: 0 4px 12px rgba(0, 0, 0, 0.3);
                background: ${type === 'success' ? 'linear-gradient(135deg, #1A4D2A, #2d5a3d)' : 'linear-gradient(135deg, #dc3545, #c82333)'};
                transform: translateX(100%);
                transition: transform 0.3s ease;
            `;
            
            document.body.appendChild(notification);
            
            // Animate in
            setTimeout(() => {
                notification.style.transform = 'translateX(0)';
            }, 100);
            
            // Auto remove after 3 seconds
            setTimeout(() => {
                notification.style.transform = 'translateX(100%)';
                setTimeout(() => {
                    if (notification.parentNode) {
                        notification.parentNode.removeChild(notification);
                    }
                }, 300);
            }, 3000);
        }

        // Update cart totals after quantity changes (fallback method)
        function updateCartTotals() {
            // Reload the page to get updated totals with proper shipping calculation
            // This is now only used as a fallback - main updates use updateSelectedItems()
            location.reload();
        }


        
        // Show custom delete confirmation dialog
        function showDeleteConfirmation(itemName, onConfirm) {
            
            // Create modal overlay
            const modalOverlay = document.createElement('div');
            modalOverlay.className = 'delete-confirmation-overlay';
            modalOverlay.style.cssText = `
                position: fixed;
                top: 0;
                left: 0;
                width: 100%;
                height: 100%;
                background: rgba(0, 0, 0, 0.6);
                display: flex;
                align-items: center;
                justify-content: center;
                z-index: 10000;
                animation: fadeIn 0.3s ease;
                padding: 20px;
                box-sizing: border-box;
            `;
            
            // Create modal content
            const modalContent = document.createElement('div');
            modalContent.className = 'delete-confirmation-modal';
            modalContent.style.cssText = `
                background: white;
                border-radius: 16px;
                padding: 25px 20px;
                max-width: 350px;
                width: 100%;
                text-align: center;
                box-shadow: 0 20px 40px rgba(0, 0, 0, 0.4);
                animation: slideIn 0.3s ease;
                position: relative;
            `;
            
            modalContent.innerHTML = `
                <div style="font-size: 60px; color: #dc3545; margin-bottom: 20px;">
                    <i class="fas fa-trash-alt"></i>
                </div>
                <h3 style="color: #1A4D2A; margin-bottom: 15px; font-size: 22px; font-weight: bold;">Remove Item?</h3>
                <p style="color: #666; margin-bottom: 30px; line-height: 1.6; font-size: 16px;">
                    Are you sure you want to remove<br>
                    <strong style="color: #1A4D2A; font-size: 17px;">"${itemName}"</strong><br>
                    from your cart?
                </p>
                <div style="display: flex; gap: 12px; justify-content: center; flex-direction: column;">
                    <button class="confirm-btn" style="
                        background: linear-gradient(135deg, #dc3545, #c82333);
                        color: white;
                        border: none;
                        padding: 16px 24px;
                        border-radius: 12px;
                        cursor: pointer;
                        font-weight: 600;
                        font-size: 16px;
                        transition: all 0.3s ease;
                        box-shadow: 0 4px 12px rgba(220, 53, 69, 0.3);
                        min-height: 48px;
                    ">Remove from Cart</button>
                    <button class="cancel-btn" style="
                        background: #f8f9fa;
                        color: #6c757d;
                        border: 2px solid #dee2e6;
                        padding: 16px 24px;
                        border-radius: 12px;
                        cursor: pointer;
                        font-weight: 600;
                        font-size: 16px;
                        transition: all 0.3s ease;
                        min-height: 48px;
                    ">Cancel</button>
                </div>
            `;
            
            modalOverlay.appendChild(modalContent);
            document.body.appendChild(modalOverlay);
            
            // Prevent body scrolling when modal is open
            document.body.style.overflow = 'hidden';
            
            // Add event listeners
            const cancelBtn = modalContent.querySelector('.cancel-btn');
            const confirmBtn = modalContent.querySelector('.confirm-btn');
            
            const closeModal = () => {
                // Restore body scrolling
                document.body.style.overflow = '';
                
                modalOverlay.style.animation = 'fadeOut 0.3s ease';
                setTimeout(() => {
                    if (modalOverlay.parentNode) {
                        modalOverlay.parentNode.removeChild(modalOverlay);
                    }
                }, 300);
            };
            
            cancelBtn.addEventListener('click', closeModal);
            confirmBtn.addEventListener('click', () => {
                closeModal();
                onConfirm();
            });
            
            modalOverlay.addEventListener('click', (e) => {
                if (e.target === modalOverlay) {
                    closeModal();
                }
            });
            
            // Add keyboard support (ESC to cancel)
            const handleKeyPress = (e) => {
                if (e.key === 'Escape') {
                    closeModal();
                    document.removeEventListener('keydown', handleKeyPress);
                }
            };
            document.addEventListener('keydown', handleKeyPress);
            
            // Add CSS animations
            if (!document.getElementById('delete-confirmation-styles')) {
                const style = document.createElement('style');
                style.id = 'delete-confirmation-styles';
                style.textContent = `
                    @keyframes fadeIn {
                        from { opacity: 0; }
                        to { opacity: 1; }
                    }
                    @keyframes fadeOut {
                        from { opacity: 1; }
                        to { opacity: 0; }
                    }
                    @keyframes slideIn {
                        from { transform: translateY(30px) scale(0.95); opacity: 0; }
                        to { transform: translateY(0) scale(1); opacity: 1; }
                    }
                    .cancel-btn:active {
                        background: #e9ecef !important;
                        transform: scale(0.98);
                    }
                    .confirm-btn:active {
                        background: linear-gradient(135deg, #c82333, #bd2130) !important;
                        transform: scale(0.98);
                    }
                    .cancel-btn:hover {
                        background: #e9ecef !important;
                    }
                    .confirm-btn:hover {
                        background: linear-gradient(135deg, #c82333, #bd2130) !important;
                    }
                `;
                document.head.appendChild(style);
            }
        }
        
        // Proceed with actual deletion
        function proceedWithDeletion(cartItemId) {
            fetch('api/cart.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'remove',
                    cart_item_id: cartItemId
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    const cartItem = document.querySelector(`[data-cart-id="${cartItemId}"]`);
                    if (cartItem) {
                        cartItem.remove();
                    }
                    // Update totals and shipping without page reload
                    updateSelectedItems();
                    // Show success notification
                    showNotification('Item removed from cart', 'success');
                } else {
                    showNotification(data.message || 'Failed to remove item from cart', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error removing from cart', 'error');
            });
        }

        // Item selection functionality
        function updateSelectedItems() {
            // Show loading indicator
            showLoading();
            
            // Add a small delay to show the loading animation
            setTimeout(() => {
                const checkboxes = document.querySelectorAll('.item-select-checkbox');
                const selectAllCheckbox = document.getElementById('select-all-items');
                const selectedCount = document.getElementById('selected-count');
                const checkoutBtn = document.getElementById('checkout-btn');
                
                // Check if elements exist before proceeding
                if (checkboxes.length === 0) {
                    hideLoading();
                    return; // No items to process
                }
                
                let selectedItems = 0;
                let selectedSubtotal = 0;
                let selectedQuantity = 0;
                
                checkboxes.forEach(checkbox => {
                    const cartItem = checkbox.closest('.cart-item');
                    
                    if (checkbox.checked) {
                        selectedItems++;
                        const itemSubtotal = parseFloat(checkbox.dataset.price) * parseInt(checkbox.dataset.quantity);
                        selectedSubtotal += itemSubtotal;
                        selectedQuantity += parseInt(checkbox.dataset.quantity);
                        cartItem.classList.remove('unselected');
                    } else {
                        cartItem.classList.add('unselected');
                    }
                });
                
                // Update select all checkbox state (only if it exists)
                if (selectAllCheckbox) {
                    if (selectedItems === checkboxes.length) {
                        selectAllCheckbox.checked = true;
                        selectAllCheckbox.indeterminate = false;
                    } else if (selectedItems === 0) {
                        selectAllCheckbox.checked = false;
                        selectAllCheckbox.indeterminate = false;
                    } else {
                        selectAllCheckbox.checked = false;
                        selectAllCheckbox.indeterminate = true;
                    }
                }
                
                // Update selected count (only if it exists)
                if (selectedCount) {
                    selectedCount.textContent = selectedItems;
                }
                
                // Update totals
                updateCartTotalsForSelected(selectedSubtotal, selectedQuantity);
                
                // Enable/disable checkout button (only if it exists)
                if (checkoutBtn) {
                    if (selectedItems === 0) {
                        checkoutBtn.disabled = true;
                        checkoutBtn.textContent = 'Select items to checkout';
                        checkoutBtn.style.opacity = '0.6';
                    } else {
                        checkoutBtn.disabled = false;
                        checkoutBtn.textContent = 'Proceed to Checkout';
                        checkoutBtn.style.opacity = '1';
                    }
                }
                
                // Hide loading indicator
                hideLoading();
            }, 300); // 300ms delay to show loading animation
        }
        
        function toggleSelectAll() {
            const selectAllCheckbox = document.getElementById('select-all-items');
            const checkboxes = document.querySelectorAll('.item-select-checkbox');
            
            // Check if elements exist
            if (!selectAllCheckbox || checkboxes.length === 0) {
                return;
            }
            
            checkboxes.forEach(checkbox => {
                checkbox.checked = selectAllCheckbox.checked;
            });
            
            updateSelectedItems();
        }
        
        function updateCartTotalsForSelected(subtotal, quantity) {
            // Calculate shipping based on current subtotal (dynamic calculation)
            // Perfume bottles are very light, so we'll use a fixed weight of 0.1kg for the entire order
            const weight = 0.1; // Fixed weight for perfume orders
            
            let shippingCost = 0;
            let shippingMethod = 'Standard Shipping';
            let deliveryTime = '2-3 days';
            
            // Use shipping configuration from server if available
            if (shippingConfig) {
                console.log('Using shipping config:', shippingConfig);
                console.log('Subtotal:', subtotal, 'Weight:', weight, '(not used in calculation)');
                console.log('Fee type check:', shippingConfig.fee_type, '->', shippingConfig.fee_type ? shippingConfig.fee_type.toLowerCase() : 'null');
                
                // Check if order qualifies for free shipping threshold
                if (shippingConfig.fee_type && shippingConfig.fee_type.toLowerCase() === 'free_shipping_threshold' && subtotal >= shippingConfig.base_fee) {
                    shippingCost = 0;
                    shippingMethod = 'Free Shipping';
                    deliveryTime = shippingConfig.delivery_days_min + '-' + shippingConfig.delivery_days_max + ' days';
                    console.log('Free shipping applied');
                } else if (shippingConfig.fee_type && shippingConfig.fee_type.toLowerCase() === 'weight_based') {
                    // Use base fee only (no weight calculation)
                    shippingCost = shippingConfig.base_fee;
                    shippingMethod = shippingConfig.name || 'Standard Shipping';
                    deliveryTime = shippingConfig.delivery_days_min + '-' + shippingConfig.delivery_days_max + ' days';
                    console.log('Base fee only calculation:', 'base_fee:', shippingConfig.base_fee, 'shippingCost:', shippingCost);
                } else {
                    // Fallback to default calculation
                    shippingCost = 5.0;
                    shippingMethod = 'Standard Shipping';
                    deliveryTime = '2-3 days';
                    console.log('Fallback to default calculation');
                }
            } else {
                // Fallback if no shipping config available
                if (subtotal >= 100) {
                    // Free shipping for orders RM 100+
                    shippingCost = 0;
                    shippingMethod = 'Free Shipping';
                    deliveryTime = '2-3 days';
                } else {
                    // Default rates
                    const baseFee = 5.0; // Default base fee
                    const perKgFee = 2.50; // Default per kg fee
                    shippingCost = baseFee + (weight * perKgFee);
                    shippingMethod = 'Standard Shipping';
                    deliveryTime = '2-3 days';
                }
            }
            
            const total = subtotal + shippingCost;
            
            // Update the summary display (only if elements exist)
            const subtotalEl = document.getElementById('subtotal');
            const shippingCostEl = document.getElementById('shipping-cost');
            const shippingMethodEl = document.getElementById('shipping-method');
            const deliveryTimeEl = document.getElementById('delivery-time');
            const totalEl = document.getElementById('total');
            
            if (subtotalEl) {
                subtotalEl.textContent = 'RM ' + subtotal.toFixed(2);
            }
            if (shippingCostEl) {
                shippingCostEl.textContent = shippingCost > 0 ? 'RM ' + shippingCost.toFixed(2) : 'Free';
            }
            if (shippingMethodEl) {
                shippingMethodEl.textContent = shippingMethod;
            }
            if (deliveryTimeEl) {
                deliveryTimeEl.textContent = deliveryTime;
            }
            if (totalEl) {
                totalEl.textContent = 'RM ' + total.toFixed(2);
            }
        }
        
        // New quantity button functionality using same logic as checkboxes
        function initializeQuantityButtons() {
            // Add event listeners to all plus/minus buttons
            document.querySelectorAll('.plus-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const cartItemId = this.dataset.cartId;
                    const cartItem = document.querySelector(`[data-cart-id="${cartItemId}"]`);
                    const quantitySpan = cartItem.querySelector('.quantity');
                    const currentQuantity = parseInt(quantitySpan.textContent) || 0;
                    const newQuantity = currentQuantity + 1;
                    
                    updateQuantityInCart(cartItemId, newQuantity);
                });
            });
            
            document.querySelectorAll('.minus-btn').forEach(button => {
                button.addEventListener('click', function() {
                    const cartItemId = this.dataset.cartId;
                    const cartItem = document.querySelector(`[data-cart-id="${cartItemId}"]`);
                    const quantitySpan = cartItem.querySelector('.quantity');
                    const currentQuantity = parseInt(quantitySpan.textContent) || 0;
                    
                    // Don't allow quantity to go below 1
                    if (currentQuantity <= 1) {
                        // Show a brief visual feedback
                        this.style.transform = 'scale(0.95)';
                        setTimeout(() => {
                            this.style.transform = '';
                        }, 150);
                        return; // Do nothing if quantity is already 1 or less
                    }
                    
                    const newQuantity = currentQuantity - 1;
                    updateQuantityInCart(cartItemId, newQuantity);
                });
            });
            
            // Update button states on page load
            updateQuantityButtonStates();
        }
        
        // Update the visual state of quantity buttons
        function updateQuantityButtonStates() {
            document.querySelectorAll('.cart-item').forEach(cartItem => {
                const quantitySpan = cartItem.querySelector('.quantity');
                const minusBtn = cartItem.querySelector('.minus-btn');
                const currentQuantity = parseInt(quantitySpan.textContent) || 0;
                
                if (minusBtn) {
                    if (currentQuantity <= 1) {
                        minusBtn.classList.add('disabled');
                        minusBtn.style.opacity = '0.5';
                        minusBtn.style.cursor = 'not-allowed';
                    } else {
                        minusBtn.classList.remove('disabled');
                        minusBtn.style.opacity = '1';
                        minusBtn.style.cursor = 'pointer';
                    }
                }
            });
        }
        
        // Update quantity in cart using same logic as checkbox calculations
        function updateQuantityInCart(cartItemId, newQuantity) {
            if (newQuantity < 0) return;
            
            // Show loading indicator
            showLoading();
            
            fetch('api/cart.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    action: 'update_quantity',
                    cart_item_id: cartItemId,
                    quantity: newQuantity
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (newQuantity === 0) {
                        // Remove the item from display
                        const cartItem = document.querySelector(`[data-cart-id="${cartItemId}"]`);
                        if (cartItem) {
                            cartItem.remove();
                        }
                    } else {
                        // Update the quantity display and data attributes
                        const cartItem = document.querySelector(`[data-cart-id="${cartItemId}"]`);
                        if (cartItem) {
                            const quantitySpan = cartItem.querySelector('.quantity');
                            const checkbox = cartItem.querySelector('.item-select-checkbox');
                            
                            if (quantitySpan) {
                                quantitySpan.textContent = newQuantity;
                            }
                            if (checkbox) {
                                checkbox.dataset.quantity = newQuantity;
                            }
                        }
                    }
                    
                    // Use the same calculation logic as checkboxes
                    updateSelectedItems();
                    
                    // Update button states after quantity change
                    updateQuantityButtonStates();
                } else {
                    showNotification(data.message || 'Failed to update cart', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Error updating cart', 'error');
            })
            .finally(() => {
                // Hide loading indicator
                hideLoading();
            });
        }
        
        function proceedToCheckout() {
            const selectedCheckboxes = document.querySelectorAll('.item-select-checkbox:checked');
            
            if (selectedCheckboxes.length === 0) {
                showNotification('Please select at least one item to checkout', 'error');
                return;
            }
            
            // Get selected cart item IDs
            const selectedItems = Array.from(selectedCheckboxes).map(checkbox => checkbox.dataset.cartId);
            
            // Pass selected items as URL parameter
            const selectedItemsParam = encodeURIComponent(JSON.stringify(selectedItems));
            window.location.href = 'checkout.php?selected_items=' + selectedItemsParam;
        }
        
        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Override the removeFromCart function from script.js
            window.removeFromCart = function(cartItemId) {
                // Get the item name for better confirmation message
                const cartItem = document.querySelector(`[data-cart-id="${cartItemId}"]`);
                const itemName = cartItem ? cartItem.querySelector('.cart-item-name').textContent : 'this item';
                
                showDeleteConfirmation(itemName, () => {
                    // Proceed with deletion
                    proceedWithDeletion(cartItemId);
                });
            };
            
            // Initialize quantity buttons
            initializeQuantityButtons();
            
            // Force update the totals immediately to override server-side values
            setTimeout(() => {
                updateSelectedItems();
            }, 100);
            
            // Also update after a longer delay to ensure it overrides
            setTimeout(() => {
                updateSelectedItems();
            }, 500);
        });
    </script>
</body>
</html>