<?php
session_start();
require_once 'models/User.php';
require_once 'includes/remember-me.php';

// Check for remember me token and auto-login
checkRememberMe();

// Check if user is logged in
if (!isset($_SESSION['user_id']) || $_SESSION['user_id'] === 'guest') {
    header('Location: login.php');
    exit();
}

// Get user data
$user = new User();
$user_data = $user->getById($_SESSION['user_id']);

if (!$user_data) {
    header('Location: login.php');
    exit();
}

$error_message = '';
$success_message = '';

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $current_password = trim($_POST['current_password'] ?? '');
    $new_password = trim($_POST['new_password'] ?? '');
    $confirm_password = trim($_POST['confirm_password'] ?? '');
    
    if (empty($current_password) || empty($new_password) || empty($confirm_password)) {
        $error_message = 'All fields are required.';
    } elseif ($new_password !== $confirm_password) {
        $error_message = 'New password and confirm password do not match.';
    } elseif (strlen($new_password) < 6) {
        $error_message = 'New password must be at least 6 characters long.';
    } else {
        // Verify current password
        if (password_verify($current_password, $user_data['password'])) {
            if ($user->changePassword($_SESSION['user_id'], $new_password)) {
                $success_message = 'Password changed successfully!';
            } else {
                $error_message = 'Failed to change password. Please try again.';
            }
        } else {
            $error_message = 'Current password is incorrect.';
        }
    }
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Change Password - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
    <style>
        .change-password-container {
            min-height: 100vh;
            background: linear-gradient(135deg, #f8f9fa 0%, #e9ecef 100%);
            padding-bottom: 100px;
        }

        .change-password-header {
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            padding: 20px 20px 30px 20px;
            text-align: center;
            position: relative;
            overflow: hidden;
        }

        .change-password-header::before {
            content: '';
            position: absolute;
            bottom: -20px;
            left: 0;
            right: 0;
            height: 40px;
            background: linear-gradient(135deg, #1A4D2A 0%, #2d5a3d 50%, #1A4D2A 100%);
            border-radius: 50% 50% 0 0 / 100% 100% 0 0;
        }

        .change-password-title {
            color: #D4AF37;
            font-size: 24px;
            font-weight: bold;
            margin-bottom: 5px;
        }

        .change-password-subtitle {
            color: #F4D03F;
            font-size: 14px;
        }

        .change-password-content {
            padding: 30px 20px;
            margin-top: -20px;
            position: relative;
            z-index: 2;
        }

        .form-container {
            background: white;
            border-radius: 15px;
            padding: 25px;
            margin-bottom: 20px;
            box-shadow: 0 2px 10px rgba(0,0,0,0.1);
        }

        .form-group {
            margin-bottom: 20px;
            position: relative;
        }

        .form-group label {
            display: block;
            margin-bottom: 8px;
            color: #1A4D2A;
            font-weight: 600;
            font-size: 14px;
        }

        .form-group input {
            width: 100%;
            padding: 12px 16px;
            padding-right: 45px;
            border: 2px solid #e9ecef;
            border-radius: 10px;
            font-size: 16px;
            background: #f8f9fa;
            color: #1A4D2A;
            transition: all 0.3s ease;
            box-sizing: border-box;
        }

        .form-group input:focus {
            outline: none;
            border-color: #D4AF37;
            background: white;
            box-shadow: 0 0 0 3px rgba(212, 175, 55, 0.1);
        }

        .password-toggle {
            position: absolute;
            right: 15px;
            top: 50%;
            transform: translateY(-50%);
            color: #666;
            cursor: pointer;
            font-size: 16px;
            transition: color 0.3s ease;
        }

        .password-toggle:hover {
            color: #D4AF37;
        }

        .password-strength {
            margin-top: 8px;
            font-size: 12px;
        }

        .strength-weak { color: #dc3545; }
        .strength-medium { color: #ffc107; }
        .strength-strong { color: #28a745; }

        .password-requirements {
            background: #f8f9fa;
            border-radius: 8px;
            padding: 15px;
            margin-top: 15px;
            font-size: 13px;
            color: #666;
        }

        .password-requirements h4 {
            margin: 0 0 10px 0;
            color: #1A4D2A;
            font-size: 14px;
        }

        .password-requirements ul {
            margin: 0;
            padding-left: 20px;
        }

        .password-requirements li {
            margin-bottom: 5px;
        }

        .requirement-met {
            color: #28a745;
        }

        .requirement-unmet {
            color: #dc3545;
        }

        .btn-change {
            width: 100%;
            padding: 15px;
            background: linear-gradient(135deg, #D4AF37, #F4D03F);
            border: none;
            border-radius: 12px;
            color: #1A4D2A;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
        }

        .btn-change:hover {
            background: linear-gradient(135deg, #1A4D2A, #2d5a3d);
            color: #D4AF37;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(26, 77, 42, 0.3);
        }

        .btn-change:disabled {
            opacity: 0.6;
            cursor: not-allowed;
            transform: none;
        }

        .btn-cancel {
            width: 100%;
            padding: 15px;
            background: transparent;
            border: 2px solid #D4AF37;
            border-radius: 12px;
            color: #D4AF37;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            display: flex;
            align-items: center;
            justify-content: center;
            gap: 8px;
            margin-top: 10px;
        }

        .btn-cancel:hover {
            background: #D4AF37;
            color: white;
        }

        .alert {
            padding: 12px 16px;
            border-radius: 8px;
            margin-bottom: 20px;
            display: flex;
            align-items: center;
            gap: 10px;
            font-size: 14px;
            font-weight: 500;
        }

        .alert-error {
            background: #fee;
            color: #c53030;
            border: 1px solid #feb2b2;
        }

        .alert-success {
            background: #f0fff4;
            color: #2f855a;
            border: 1px solid #9ae6b4;
        }

        .alert i {
            font-size: 16px;
        }

        @media (max-width: 480px) {
            .change-password-content {
                padding: 20px 15px;
            }
            
            .form-container {
                padding: 20px;
            }
        }
    </style>
</head>
<body>
    <div class="change-password-container">
        <!-- Top Navigation -->
        <nav class="top-nav">
            <div class="nav-left">
                <i class="fas fa-arrow-left" onclick="goBack()"></i>
            </div>
            <div class="nav-center">
                <div class="logo">
                    <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
                </div>
            </div>
            <div class="nav-right">
                <i class="fas fa-search" onclick="window.location.href='products.php'"></i>
            </div>
        </nav>

        <!-- Change Password Header -->
        <div class="change-password-header">
            <div class="change-password-title">Change Password</div>
            <div class="change-password-subtitle">Update your account security</div>
        </div>

        <!-- Change Password Content -->
        <div class="change-password-content">
            <div class="form-container">
                <?php if ($error_message): ?>
                    <div class="alert alert-error">
                        <i class="fas fa-exclamation-circle"></i>
                        <?php echo htmlspecialchars($error_message); ?>
                    </div>
                <?php endif; ?>

                <?php if ($success_message): ?>
                    <div class="alert alert-success">
                        <i class="fas fa-check-circle"></i>
                        <?php echo htmlspecialchars($success_message); ?>
                    </div>
                <?php endif; ?>

                <form method="POST" action="" id="changePasswordForm">
                    <div class="form-group">
                        <label for="current_password">Current Password *</label>
                        <input type="password" id="current_password" name="current_password" required>
                        <i class="fas fa-eye password-toggle" onclick="togglePassword('current_password')"></i>
                    </div>

                    <div class="form-group">
                        <label for="new_password">New Password *</label>
                        <input type="password" id="new_password" name="new_password" required oninput="checkPasswordStrength()">
                        <i class="fas fa-eye password-toggle" onclick="togglePassword('new_password')"></i>
                        <div id="password-strength" class="password-strength"></div>
                    </div>

                    <div class="form-group">
                        <label for="confirm_password">Confirm New Password *</label>
                        <input type="password" id="confirm_password" name="confirm_password" required oninput="checkPasswordMatch()">
                        <i class="fas fa-eye password-toggle" onclick="togglePassword('confirm_password')"></i>
                        <div id="password-match" class="password-strength"></div>
                    </div>

                    <div class="password-requirements">
                        <h4>Password Requirements:</h4>
                        <ul>
                            <li id="req-length" class="requirement-unmet">At least 6 characters long</li>
                            <li id="req-uppercase" class="requirement-unmet">Contains uppercase letter</li>
                            <li id="req-lowercase" class="requirement-unmet">Contains lowercase letter</li>
                            <li id="req-number" class="requirement-unmet">Contains number</li>
                        </ul>
                    </div>

                    <button type="submit" class="btn-change" id="changeBtn">
                        <i class="fas fa-lock"></i>
                        Change Password
                    </button>
                </form>

                <button class="btn-cancel" onclick="goBack()">
                    <i class="fas fa-times"></i>
                    Cancel
                </button>
            </div>
        </div>

        <!-- Bottom Navigation -->
        <nav class="bottom-nav">
            <div class="nav-item" onclick="window.location.href='index.php'">
                <i class="fas fa-home"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='products.php'">
                <i class="fas fa-box"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='cart.php'">
                <i class="fas fa-shopping-cart"></i>
            </div>
            <div class="nav-item" onclick="window.location.href='profile.php'">
                <i class="far fa-user"></i>
            </div>
        </nav>
    </div>

    <script>
        function goBack() {
            window.history.back();
        }

        function togglePassword(fieldId) {
            const field = document.getElementById(fieldId);
            const icon = field.nextElementSibling;
            
            if (field.type === 'password') {
                field.type = 'text';
                icon.classList.remove('fa-eye');
                icon.classList.add('fa-eye-slash');
            } else {
                field.type = 'password';
                icon.classList.remove('fa-eye-slash');
                icon.classList.add('fa-eye');
            }
        }

        function checkPasswordStrength() {
            const password = document.getElementById('new_password').value;
            const strengthDiv = document.getElementById('password-strength');
            const changeBtn = document.getElementById('changeBtn');
            
            let strength = 0;
            let strengthText = '';
            let strengthClass = '';
            
            // Check requirements
            const hasLength = password.length >= 6;
            const hasUppercase = /[A-Z]/.test(password);
            const hasLowercase = /[a-z]/.test(password);
            const hasNumber = /\d/.test(password);
            
            // Update requirement indicators
            document.getElementById('req-length').className = hasLength ? 'requirement-met' : 'requirement-unmet';
            document.getElementById('req-uppercase').className = hasUppercase ? 'requirement-met' : 'requirement-unmet';
            document.getElementById('req-lowercase').className = hasLowercase ? 'requirement-met' : 'requirement-unmet';
            document.getElementById('req-number').className = hasNumber ? 'requirement-met' : 'requirement-unmet';
            
            // Calculate strength
            if (hasLength) strength++;
            if (hasUppercase) strength++;
            if (hasLowercase) strength++;
            if (hasNumber) strength++;
            
            if (strength === 0) {
                strengthText = '';
                strengthClass = '';
            } else if (strength <= 2) {
                strengthText = 'Weak';
                strengthClass = 'strength-weak';
            } else if (strength === 3) {
                strengthText = 'Medium';
                strengthClass = 'strength-medium';
            } else {
                strengthText = 'Strong';
                strengthClass = 'strength-strong';
            }
            
            strengthDiv.textContent = strengthText;
            strengthDiv.className = 'password-strength ' + strengthClass;
            
            // Enable/disable submit button
            const allRequirementsMet = hasLength && hasUppercase && hasLowercase && hasNumber;
            changeBtn.disabled = !allRequirementsMet;
        }

        function checkPasswordMatch() {
            const newPassword = document.getElementById('new_password').value;
            const confirmPassword = document.getElementById('confirm_password').value;
            const matchDiv = document.getElementById('password-match');
            const changeBtn = document.getElementById('changeBtn');
            
            if (confirmPassword === '') {
                matchDiv.textContent = '';
                matchDiv.className = 'password-strength';
            } else if (newPassword === confirmPassword) {
                matchDiv.textContent = 'Passwords match';
                matchDiv.className = 'password-strength strength-strong';
            } else {
                matchDiv.textContent = 'Passwords do not match';
                matchDiv.className = 'password-strength strength-weak';
            }
            
            // Re-check if submit button should be enabled
            checkPasswordStrength();
        }

        // Initialize cart badge
        document.addEventListener('DOMContentLoaded', function() {
            updateCartBadge();
        });
    </script>
</body>
</html>