<?php
/**
 * Complete Analytics Debug & Fix Tool
 * Tests everything and shows you what's working
 */

require_once 'config/database.php';

echo "<h2>🔧 Analytics System Debug Tool</h2>";
echo "<style>
    body { font-family: Arial, sans-serif; padding: 20px; background: #f5f5f5; }
    .success { background: #d4edda; padding: 15px; border-left: 4px solid #28a745; margin: 10px 0; }
    .error { background: #f8d7da; padding: 15px; border-left: 4px solid #dc3545; margin: 10px 0; }
    .warning { background: #fff3cd; padding: 15px; border-left: 4px solid #ffc107; margin: 10px 0; }
    .info { background: #d1ecf1; padding: 15px; border-left: 4px solid #0c5460; margin: 10px 0; }
    table { border-collapse: collapse; width: 100%; margin: 10px 0; background: white; }
    th, td { border: 1px solid #ddd; padding: 12px; text-align: left; }
    th { background-color: #4CAF50; color: white; }
    button { padding: 10px 20px; background: #007bff; color: white; border: none; border-radius: 4px; cursor: pointer; margin: 5px; }
    button:hover { background: #0056b3; }
    .fix-btn { background: #28a745; }
    .fix-btn:hover { background: #218838; }
    pre { background: #f5f5f5; padding: 10px; border-radius: 4px; overflow-x: auto; }
    .section { background: white; padding: 20px; margin: 20px 0; border-radius: 8px; box-shadow: 0 2px 4px rgba(0,0,0,0.1); }
</style>";

try {
    $conn = getDB();
    
    // ============================================
    // 1. CHECK TABLE STRUCTURE
    // ============================================
    echo "<div class='section'>";
    echo "<h3>📋 Step 1: Database Table Structure</h3>";
    
    $stmt = $conn->query("SHOW TABLES LIKE 'product_analytics_simple'");
    if ($stmt->rowCount() == 0) {
        echo "<div class='error'>";
        echo "<h4>❌ Table Doesn't Exist!</h4>";
        echo "<p><strong>Fix:</strong> Run <a href='setup-simple-analytics.php'>setup-simple-analytics.php</a></p>";
        echo "</div>";
        echo "</div>";
        exit;
    }
    
    // Check columns
    $stmt = $conn->query("DESCRIBE product_analytics_simple");
    $columns = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    $hasClicks = false;
    $hasFavorites = false;
    $hasViews = false;
    
    echo "<table>";
    echo "<tr><th>Column Name</th><th>Type</th><th>Status</th></tr>";
    foreach ($columns as $col) {
        $status = "✅ OK";
        $rowClass = "";
        
        if ($col['Field'] === 'clicks') {
            $hasClicks = true;
            $status = "⚠️ OLD - Need to rename";
            $rowClass = "style='background: #fff3cd;'";
        }
        if ($col['Field'] === 'favorites') {
            $hasFavorites = true;
        }
        if ($col['Field'] === 'views') {
            $hasViews = true;
        }
        
        echo "<tr $rowClass>";
        echo "<td><strong>{$col['Field']}</strong></td>";
        echo "<td>{$col['Type']}</td>";
        echo "<td>$status</td>";
        echo "</tr>";
    }
    echo "</table>";
    
    // Show fix if needed
    if ($hasClicks && !$hasFavorites) {
        echo "<div class='warning'>";
        echo "<h4>⚠️ Action Required: Rename Column</h4>";
        echo "<p>Your table has 'clicks' but needs 'favorites'.</p>";
        echo "<button class='fix-btn' onclick=\"runSqlFix()\">🔧 Fix Now (Rename clicks → favorites)</button>";
        echo "<p><strong>Or run this SQL manually:</strong></p>";
        echo "<pre>ALTER TABLE product_analytics_simple CHANGE COLUMN clicks favorites INT DEFAULT 0;</pre>";
        echo "</div>";
    } else if ($hasFavorites) {
        echo "<div class='success'>";
        echo "<p>✅ Table structure is correct!</p>";
        echo "</div>";
    }
    echo "</div>";
    
    // ============================================
    // 2. TEST VIEW TRACKING
    // ============================================
    echo "<div class='section'>";
    echo "<h3>👁️ Step 2: Test View Tracking</h3>";
    
    $testProductId = 22; // Change if needed
    
    // Simulate a view
    $stmt = $conn->prepare("
        INSERT INTO product_analytics_simple (product_id, views, favorites, add_to_cart)
        VALUES (?, 1, 0, 0)
        ON DUPLICATE KEY UPDATE 
            views = views + 1,
            last_viewed = NOW()
    ");
    $result = $stmt->execute([$testProductId]);
    
    if ($result) {
        echo "<div class='success'>";
        echo "<p>✅ View tracking works! Just added a test view for product #$testProductId</p>";
        echo "</div>";
        
        // Show the record
        $stmt = $conn->prepare("SELECT * FROM product_analytics_simple WHERE product_id = ?");
        $stmt->execute([$testProductId]);
        $record = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if ($record) {
            echo "<p><strong>Current stats for product #$testProductId:</strong></p>";
            echo "<table>";
            echo "<tr><th>Views</th><th>Favorites</th><th>Add to Cart</th><th>Last Viewed</th></tr>";
            echo "<tr>";
            echo "<td><strong>" . $record['views'] . "</strong></td>";
            echo "<td>" . $record['favorites'] . "</td>";
            echo "<td>" . $record['add_to_cart'] . "</td>";
            echo "<td>" . ($record['last_viewed'] ?? 'Never') . "</td>";
            echo "</tr>";
            echo "</table>";
        }
    } else {
        echo "<div class='error'>";
        echo "<p>❌ View tracking failed!</p>";
        echo "</div>";
    }
    echo "</div>";
    
    // ============================================
    // 3. TEST ENDPOINT
    // ============================================
    echo "<div class='section'>";
    echo "<h3>🔌 Step 3: Test Analytics Endpoint</h3>";
    
    $testData = [
        'event_type' => 'product_view',
        'product_id' => $testProductId,
        'user_id' => 2,
        'session_id' => 'test_' . time()
    ];
    
    // Test the endpoint - use relative path to work on both localhost and live
    $protocol = isset($_SERVER['HTTPS']) && $_SERVER['HTTPS'] === 'on' ? 'https' : 'http';
    $host = $_SERVER['HTTP_HOST'];
    $baseDir = dirname($_SERVER['PHP_SELF']);
    $endpointUrl = $protocol . '://' . $host . $baseDir . '/ajax/track-analytics-simple.php';
    
    $ch = curl_init();
    curl_setopt($ch, CURLOPT_URL, $endpointUrl);
    curl_setopt($ch, CURLOPT_POST, 1);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($testData));
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    
    $response = curl_exec($ch);
    $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    curl_close($ch);
    
    if ($httpCode == 200) {
        $responseData = json_decode($response, true);
        if (isset($responseData['success']) && $responseData['success']) {
            echo "<div class='success'>";
            echo "<p>✅ Analytics endpoint is working!</p>";
            echo "<p><strong>Response:</strong></p>";
            echo "<pre>" . json_encode($responseData, JSON_PRETTY_PRINT) . "</pre>";
            echo "</div>";
        } else {
            echo "<div class='error'>";
            echo "<p>❌ Endpoint returned error:</p>";
            echo "<pre>" . $response . "</pre>";
            echo "</div>";
        }
    } else {
        echo "<div class='error'>";
        echo "<p>❌ HTTP Error: $httpCode</p>";
        echo "<p>Response: $response</p>";
        echo "</div>";
    }
    echo "</div>";
    
    // ============================================
    // 4. CHECK JAVASCRIPT FILE
    // ============================================
    echo "<div class='section'>";
    echo "<h3>📜 Step 4: JavaScript File Check</h3>";
    
    if (file_exists('js/analytics-tracking.js')) {
        $jsContent = file_get_contents('js/analytics-tracking.js');
        
        $checks = [
            'Has endpoint' => strpos($jsContent, 'ajax/track-analytics-simple.php') !== false,
            'Has trackProductView' => strpos($jsContent, 'trackProductView') !== false,
            'Has trackAddToCart' => strpos($jsContent, 'trackAddToCart') !== false,
            'Has trackAddFavorite' => strpos($jsContent, 'trackAddFavorite') !== false,
        ];
        
        $allGood = true;
        echo "<table>";
        echo "<tr><th>Check</th><th>Status</th></tr>";
        foreach ($checks as $check => $passed) {
            echo "<tr>";
            echo "<td>$check</td>";
            echo "<td>" . ($passed ? "✅ Pass" : "❌ Fail") . "</td>";
            echo "</tr>";
            if (!$passed) $allGood = false;
        }
        echo "</table>";
        
        if ($allGood) {
            echo "<div class='success'><p>✅ JavaScript file is configured correctly!</p></div>";
        } else {
            echo "<div class='warning'><p>⚠️ Some checks failed. Make sure you ran all updates.</p></div>";
        }
    } else {
        echo "<div class='error'><p>❌ File js/analytics-tracking.js not found!</p></div>";
    }
    echo "</div>";
    
    // ============================================
    // 5. CURRENT DATA SUMMARY
    // ============================================
    echo "<div class='section'>";
    echo "<h3>📊 Step 5: Current Analytics Data</h3>";
    
    $stmt = $conn->query("
        SELECT 
            pas.product_id,
            p.name,
            pas.views,
            pas.favorites,
            pas.add_to_cart,
            pas.updated_at
        FROM product_analytics_simple pas
        LEFT JOIN products p ON pas.product_id = p.id
        WHERE pas.views > 0 OR pas.favorites > 0 OR pas.add_to_cart > 0
        ORDER BY pas.updated_at DESC
        LIMIT 10
    ");
    
    $data = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if (count($data) > 0) {
        echo "<p>✅ Found " . count($data) . " products with tracking data:</p>";
        echo "<table>";
        echo "<tr><th>Product ID</th><th>Name</th><th>Views</th><th>Favorites</th><th>Add to Cart</th><th>Updated</th></tr>";
        foreach ($data as $row) {
            echo "<tr>";
            echo "<td>{$row['product_id']}</td>";
            echo "<td>" . htmlspecialchars($row['name'] ?? 'Unknown') . "</td>";
            echo "<td><strong>{$row['views']}</strong></td>";
            echo "<td><strong>{$row['favorites']}</strong></td>";
            echo "<td><strong>{$row['add_to_cart']}</strong></td>";
            echo "<td>{$row['updated_at']}</td>";
            echo "</tr>";
        }
        echo "</table>";
    } else {
        echo "<div class='warning'>";
        echo "<p>⚠️ No tracking data found yet. This is normal for a new installation.</p>";
        echo "<p><strong>To test:</strong> Visit a product page and check back here.</p>";
        echo "</div>";
    }
    
    // Show totals
    $stmt = $conn->query("
        SELECT 
            SUM(views) as total_views,
            SUM(favorites) as total_favorites,
            SUM(add_to_cart) as total_cart
        FROM product_analytics_simple
    ");
    $totals = $stmt->fetch(PDO::FETCH_ASSOC);
    
    echo "<p><strong>Overall Totals:</strong></p>";
    echo "<ul>";
    echo "<li>Total Views: <strong>" . ($totals['total_views'] ?? 0) . "</strong></li>";
    echo "<li>Total Favorites: <strong>" . ($totals['total_favorites'] ?? 0) . "</strong></li>";
    echo "<li>Total Add to Cart: <strong>" . ($totals['total_cart'] ?? 0) . "</strong></li>";
    echo "</ul>";
    echo "</div>";
    
    // ============================================
    // FINAL SUMMARY
    // ============================================
    echo "<div class='section' style='background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white;'>";
    echo "<h3 style='color: white;'>✅ Summary & Next Steps</h3>";
    echo "<div style='background: rgba(255,255,255,0.1); padding: 15px; border-radius: 8px;'>";
    
    if ($hasFavorites && $hasViews) {
        echo "<p><strong>✅ Everything looks good!</strong></p>";
        echo "<ol style='line-height: 2;'>";
        echo "<li>Database table structure: ✅ Correct</li>";
        echo "<li>View tracking: ✅ Working</li>";
        echo "<li>Endpoint: ✅ Responding</li>";
        echo "</ol>";
        echo "<p><strong>Test it:</strong></p>";
        echo "<ol>";
        echo "<li>Visit: <a href='product-detail.php?id=$testProductId' target='_blank' style='color: white; text-decoration: underline;'>Product #$testProductId</a></li>";
        echo "<li>Add product to favorites (click ❤️)</li>";
        echo "<li>Add to cart</li>";
        echo "<li>Refresh this page to see updated stats</li>";
        echo "</ol>";
    } else {
        echo "<p><strong>⚠️ Action needed:</strong></p>";
        if ($hasClicks) {
            echo "<p>1. Run the SQL fix to rename 'clicks' → 'favorites'</p>";
        }
        echo "<p>2. Clear browser cache</p>";
        echo "<p>3. Test a product page</p>";
    }
    echo "</div>";
    echo "</div>";
    
} catch (Exception $e) {
    echo "<div class='error'>";
    echo "<h3>❌ Error:</h3>";
    echo "<p>" . $e->getMessage() . "</p>";
    echo "</div>";
}
?>

<script>
function runSqlFix() {
    if (confirm('This will rename the "clicks" column to "favorites". Continue?')) {
        fetch('ajax/run-sql-fix.php', {
            method: 'POST',
            headers: {'Content-Type': 'application/json'},
            body: JSON.stringify({action: 'rename_clicks_to_favorites'})
        })
        .then(response => response.json())
        .then(data => {
            if (data.success) {
                alert('✅ Fixed! Reloading page...');
                location.reload();
            } else {
                alert('❌ Error: ' + data.error);
            }
        })
        .catch(err => {
            alert('❌ Error: ' + err.message);
        });
    }
}
</script>
