<?php
require_once 'includes/auth-check.php';

// Require authentication for favorites
requireAuth();

require_once 'models/Product.php';
require_once 'config/database.php';

// Get user's favorite products from database
$conn = getDB();
$user_id = $_SESSION['user_id'];
$product = new Product();

// Get favorite product IDs for the current user
$stmt = $conn->prepare("SELECT product_id FROM wishlist WHERE user_id = ? ORDER BY created_at DESC");
$stmt->execute([$user_id]);
$favorite_ids = $stmt->fetchAll(PDO::FETCH_COLUMN);

$favoriteProducts = [];

if (!empty($favorite_ids)) {
    // Get product details for favorite products using Product model for consistent pricing
    $allProducts = $product->getAll(['limit' => null]);
    $favoriteProducts = array_filter($allProducts, function($p) use ($favorite_ids) {
        return in_array($p['id'], $favorite_ids);
    });
    
    // Reorder to match the order in wishlist (most recent first)
    $orderedFavorites = [];
    foreach ($favorite_ids as $id) {
        foreach ($favoriteProducts as $product) {
            if ($product['id'] == $id) {
                $orderedFavorites[] = $product;
                break;
            }
        }
    }
    $favoriteProducts = $orderedFavorites;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Favorites - 2 Days Parfume</title>
    <link rel="stylesheet" href="styles.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
    <style>
        /* Ensure body has proper spacing for fixed navigation */
        body {
            padding-bottom: 80px;
        }
        
        /* Favorites Page Specific Styles */
        .favorites-header {
            text-align: center;
            margin: 20px 0 30px 0;
            padding: 25px 20px;
            background: linear-gradient(135deg, #ffffff, #fffef7);
            border-radius: 15px;
            box-shadow: 0 2px 8px rgba(253, 216, 53, 0.15);
            border: 1px solid #FDD835;
        }

        .favorites-header h1 {
            font-size: 32px;
            font-weight: bold;
            color: #333333;
            margin-bottom: 8px;
            text-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
        }

        .favorites-subtitle {
            font-size: 16px;
            color: #666666;
            margin-bottom: 10px;
            font-weight: 400;
        }

        .favorites-count {
            font-size: 14px;
            color: #888888;
            font-weight: 500;
            background: linear-gradient(135deg, #FDD835, #FBC02D);
            color: #333333;
            padding: 6px 12px;
            border-radius: 15px;
            display: inline-block;
            box-shadow: 0 2px 6px rgba(253, 216, 53, 0.3);
        }

        .favorites-main {
            padding: 0 20px 150px 20px;
        }

        /* Empty Favorites State */
        .empty-favorites {
            text-align: center;
            padding: 60px 20px;
            background: linear-gradient(135deg, #ffffff, #fffef7);
            border-radius: 15px;
            box-shadow: 0 2px 8px rgba(253, 216, 53, 0.15);
            border: 1px solid #FDD835;
        }

        .empty-icon {
            font-size: 4rem;
            color: #FDD835;
            margin-bottom: 20px;
        }

        .empty-favorites h3 {
            font-size: 24px;
            color: #333333;
            margin-bottom: 10px;
            font-weight: bold;
        }

        .empty-favorites p {
            font-size: 16px;
            color: #666666;
            margin-bottom: 30px;
        }

        .browse-btn {
            background: linear-gradient(135deg, #FDD835, #FBC02D);
            color: #333333;
            border: 2px solid #FBC02D;
            padding: 12px 24px;
            border-radius: 25px;
            font-size: 16px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(253, 216, 53, 0.3);
        }

        .browse-btn:hover {
            background: linear-gradient(135deg, #FBC02D, #FDD835);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(253, 216, 53, 0.4);
        }

        /* Favorites Grid */
        .favorites-grid {
            display: grid;
            grid-template-columns: 1fr 1fr;
            gap: 9px;
            
        }

        .favorite-card {
            background: #ffffff;
            border-radius: 10px;
            padding: 7px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            border: 1px solid #e0e0e0;
            transition: all 0.3s ease;
            position: relative;
            overflow: hidden;
            display: flex;
            flex-direction: column;
            height: 100%;
        }

        .favorite-card:hover {
            transform: translateY(-2px);
            box-shadow: 0 4px 16px rgba(0, 0, 0, 0.15);
            border-color: #FDD835;
        }

        .favorite-image {
            height: 110px;
            position: relative;
            width: 100%;
            margin-bottom: 15px;
            display: flex;
            align-items: center;
            justify-content: center;
            background: #f5f5f5;
            border-radius: 8px;
            overflow: visible;
        }

        .favorite-image img {
            width: 100%;
            height: 100%;
            object-fit: contain;
            border-radius: 8px;
        }

        .favorite-btn {
            position: absolute;
            top: 8px;
            right: 8px;
            background: #FDD835;
            border: none;
            width: 32px;
            height: 32px;
            border-radius: 50%;
            display: flex;
            align-items: center;
            justify-content: center;
            cursor: pointer;
            font-size: 14px;
            color: #333333;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(253, 216, 53, 0.3);
            z-index: 3;
        }

        .favorite-btn.active {
            background: #FDD835;
            color: #333333;
            box-shadow: 0 2px 8px rgba(253, 216, 53, 0.4);
        }

        .favorite-btn:hover {
            transform: scale(1.1);
        }

        .favorite-info {
            text-align: left;
            flex-grow: 1;
            display: flex;
            flex-direction: column;
            justify-content: space-between;
        }

        .favorite-name {
            font-size: 13px;
            font-weight: 700;
            color: #333333;
            margin-bottom: 6px;
            line-height: 1.2;
            text-transform: uppercase;
            letter-spacing: 0.5px;
        }

        .favorite-code {
            font-size: 10px;
            color: #666666;
            margin-bottom: 10px;
            font-weight: 500;
        }

        .favorite-pricing {
            display: flex;
            align-items: center;
            gap: 10px;
            margin-bottom: 20px;
        }

        .original-price {
            font-size: 14px;
            color: #999999;
            text-decoration: line-through;
            font-weight: 500;
        }

        .current-price {
            font-size: 15px;
            font-weight: 700;
            color: #FDD835;
            text-shadow: 0 1px 2px rgba(0, 0, 0, 0.1);
        }

        .add-to-cart-btn {
            width: 100%;
            background: #FDD835;
            color: #333333;
            border: none;
            border-radius: 50px;
            padding: 8px;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s ease;
            box-shadow: 0 2px 8px rgba(253, 216, 53, 0.3);
            text-transform: uppercase;
            letter-spacing: 0.5px;
            text-align: center;
            font-size: 11px;
            line-height: 1.2;
        }

        .add-to-cart-btn:hover {
            background: #FBC02D;
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(253, 216, 53, 0.4);
        }
    </style>
</head>
<body>
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-arrow-left" onclick="goBack()"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <i class="far fa-user" onclick="window.location.href='profile.php'"></i>
        </div>
    </nav>

    <!-- Page Header -->
    <div class="favorites-header">
        <h1>Favorites</h1>
        <p class="favorites-subtitle">Your Loved Parfumes</p>
        <div class="favorites-count"><?php echo count($favoriteProducts); ?> items</div>
    </div>

    <!-- Main Content -->
    <main class="favorites-main">
        <?php if (empty($favoriteProducts)): ?>
        <!-- Empty State -->
        <div class="empty-favorites">
            <div class="empty-icon">
                <i class="far fa-heart"></i>
            </div>
            <h3>No Favorites Yet</h3>
            <p>Start adding products to your favorites!</p>
            <button class="browse-btn" onclick="window.location.href='products.php'">
                Browse Products
            </button>
        </div>
        <?php else: ?>
        <!-- Favorites Grid -->
        <div class="favorites-grid">
            <?php foreach ($favoriteProducts as $product): ?>
            <div class="favorite-card <?php echo $product['is_out_of_stock'] ? 'out-of-stock' : ''; ?>" data-product-id="<?php echo $product['id']; ?>" onclick="viewProduct(<?php echo $product['id']; ?>)">
                <div class="favorite-image">
                    <img src="<?php echo $product['image']; ?>" alt="<?php echo $product['name']; ?>" onerror="this.src='data:image/svg+xml;base64,PHN2ZyB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgeG1sbnM9Imh0dHA6Ly93d3cudzMub3JnLzIwMDAvc3ZnIj48cmVjdCB3aWR0aD0iMTAwIiBoZWlnaHQ9IjEwMCIgZmlsbD0iI2Y0ZjRmNCIvPjx0ZXh0IHg9IjUwIiB5PSI1MCIgZm9udC1mYW1pbHk9IkFyaWFsIiBmb250LXNpemU9IjE0IiBmaWxsPSIjOTk5IiB0ZXh0LWFuY2hvcj0ibWlkZGxlIj5JbWFnZTwvdGV4dD48L3N2Zz4='">
                    
                    <!-- Sold Out Badge -->
                    <?php if ($product['is_out_of_stock']): ?>
                        <div class="sold-out-badge">
                            <i class="fas fa-times-circle"></i> Sold Out
                        </div>
                    <?php endif; ?>
                    
                    <button class="favorite-btn active" onclick="event.stopPropagation(); toggleFavorite(<?php echo $product['id']; ?>, this)">
                        <i class="fas fa-heart"></i>
                    </button>
                    
                    <!-- Out of Stock Overlay -->
                    <?php if ($product['is_out_of_stock']): ?>
                        <div class="out-of-stock-overlay">
                            <span>Out of Stock</span>
                        </div>
                    <?php endif; ?>
                </div>
                <div class="favorite-info">
                    <h3 class="favorite-name"><?php echo strtoupper($product['name']); ?></h3>
                    <p class="favorite-code">Code: <?php echo strtoupper(substr($product['name'], 0, 2)) . rand(10, 99); ?></p>
                    <div class="favorite-pricing">
                        <span class="current-price">RM <?php echo number_format($product['display_price'] ?? $product['price'], 2); ?></span>
                    </div>
                    <?php if ($product['is_out_of_stock']): ?>
                        <button class="add-to-cart-btn" disabled style="background: #6c757d; cursor: not-allowed;">
                            Sold Out
                        </button>
                    <?php else: ?>
                        <button class="add-to-cart-btn" onclick="event.stopPropagation(); addToCart({
                            id: <?php echo $product['id']; ?>,
                            name: '<?php echo htmlspecialchars($product['name']); ?>',
                            price: '<?php echo $product['display_price'] ?? $product['price']; ?>',
                            image: '<?php echo $product['image']; ?>',
                            qty: 1
                        })">
                            Add to Cart
                        </button>
                    <?php endif; ?>
                </div>
            </div>
            <?php endforeach; ?>
        </div>
        <?php endif; ?>
    </main>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='products.php'">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='cart.php'">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item active">
            <i class="far fa-heart"></i>
        </div>
    </nav>

    <script>
        function goBack() {
            window.history.back();
        }

        function viewProduct(productId) {
            window.location.href = 'product-detail.php?id=' + productId;
        }

        function toggleFavorite(productId, button) {
            const card = button.closest('.favorite-card');
            const isActive = button.classList.contains('active');
            const action = isActive ? 'remove' : 'add';
            
            // Show loading state
            button.disabled = true;
            const originalContent = button.innerHTML;
            button.innerHTML = '<i class="fas fa-spinner fa-spin"></i>';
            
            // Send request to server
            fetch('ajax/toggle-favorite.php', {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    product_id: productId,
                    action: action
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    if (action === 'remove') {
                        // Remove from favorites
                        button.classList.remove('active');
                        button.innerHTML = '<i class="far fa-heart"></i>';
                        card.style.opacity = '0.5';
                        card.style.transform = 'scale(0.95)';
                        
                        // Animate removal
                        setTimeout(() => {
                            card.remove();
                            updateFavoritesCount();
                        }, 300);
                    } else {
                        // Add to favorites
                        button.classList.add('active');
                        button.innerHTML = '<i class="fas fa-heart"></i>';
                    }
                    
                    // Update count
                    updateFavoritesCount();
                } else {
                    // Restore button state on error
                    button.innerHTML = originalContent;
                    showNotification(data.message || 'Error updating favorites', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                button.innerHTML = originalContent;
                showNotification('Error updating favorites', 'error');
            })
            .finally(() => {
                button.disabled = false;
            });
        }

        function addToCart(product) {
            // Use the global addToCart function from script.js
            if (window.PerfumeApp && typeof window.PerfumeApp.addToCart === 'function') {
                window.PerfumeApp.addToCart(product);
            } else if (typeof addToCart === 'function') {
                addToCart(product);
            } else {
                console.error('addToCart function not available');
                alert('Unable to add item to cart. Please try again.');
            }
        }

        function updateFavoritesCount() {
            // Get count from server
            fetch('ajax/get-favorites.php')
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        const countElement = document.querySelector('.favorites-count');
                        if (countElement) {
                            countElement.textContent = data.count + ' items';
                        }
                    }
                })
                .catch(error => {
                    console.error('Error updating favorites count:', error);
                });
        }

        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.style.cssText = `
                position: fixed;
                top: 20px;
                left: 50%;
                transform: translateX(-50%);
                background: ${type === 'info' ? '#1A4D2A' : type === 'success' ? '#28a745' : '#dc3545'};
                color: white;
                padding: 15px 20px;
                border-radius: 8px;
                font-size: 14px;
                z-index: 10000;
                box-shadow: 0 4px 12px rgba(0,0,0,0.2);
                max-width: 90%;
                text-align: center;
            `;
            notification.textContent = message;
            document.body.appendChild(notification);

            setTimeout(() => {
                if (notification.parentNode) {
                    notification.parentNode.removeChild(notification);
                }
            }, 3000);
        }

        // Load favorites count on page load
        document.addEventListener('DOMContentLoaded', function() {
            updateFavoritesCount();
        });
    </script>
</body>
</html>