<?php
require_once __DIR__ . '/../config/database.php';

class Cart {
    private $conn;
    private $table_name = "cart";

    public function __construct() {
        $this->conn = getDB();
    }

    /**
     * Add item to cart
     */
    public function addItem($user_id, $session_id, $product_id, $quantity, $price, $size = null, $version = null) {
        // Debug logging
        error_log("Cart::addItem - user_id: $user_id, session_id: $session_id, product_id: $product_id, size: $size, version: $version, quantity: $quantity, price: $price");
        
        // Check if item already exists for current user
        $existing = $this->getCartItem($user_id, $session_id, $product_id, $size, $version);
        
        if ($existing) {
            error_log("Cart::addItem - Found existing item: ID {$existing['id']}, current quantity: {$existing['quantity']}, new quantity: " . ($existing['quantity'] + $quantity));
            // Update quantity
            $this->updateQuantity($existing['id'], $existing['quantity'] + $quantity);
            return $existing['id']; // Return existing cart item ID
        } else {
            // Check if item exists for same session but different user (guest to logged-in user transition)
            $sessionItem = $this->getSessionCartItem($session_id, $product_id, $size, $version);
            if ($sessionItem) {
                error_log("Cart::addItem - Found session item: ID {$sessionItem['id']}, transferring to user $user_id");
                // Transfer ownership to current user and update quantity
                $this->transferCartItem($sessionItem['id'], $user_id, $sessionItem['quantity'] + $quantity);
                return $sessionItem['id'];
            }
            
            error_log("Cart::addItem - No existing item found, creating new item");
            // Add new item
            $query = "INSERT INTO " . $this->table_name . " 
                      (user_id, session_id, product_id, quantity, size, version, price) 
                      VALUES (:user_id, :session_id, :product_id, :quantity, :size, :version, :price)";
            
            $stmt = $this->conn->prepare($query);
            $stmt->bindValue(':user_id', $user_id ?: null);
            $stmt->bindValue(':session_id', $session_id ?: null);
            $stmt->bindValue(':product_id', $product_id, PDO::PARAM_INT);
            $stmt->bindValue(':quantity', $quantity, PDO::PARAM_INT);
            $stmt->bindValue(':size', $size ?: '');
            $stmt->bindValue(':version', $version ?: '');
            $stmt->bindValue(':price', $price);
            
            if ($stmt->execute()) {
                $newId = $this->conn->lastInsertId();
                error_log("Cart::addItem - Successfully created new item with ID: $newId");
                return $newId; // Return new cart item ID
            }
            error_log("Cart::addItem - Failed to execute INSERT query");
            return false;
        }
    }

    /**
     * Get cart items
     */
    public function getCartItems($user_id = null, $session_id = null) {
        $query = "SELECT c.*, p.name, p.brand, p.image, p.currency, p.point_rewards 
                  FROM " . $this->table_name . " c 
                  JOIN products p ON c.product_id = p.id 
                  WHERE 1=1";
        
        $params = [];
        
        if ($user_id) {
            // For logged-in users, only check user_id (not session_id)
            // This ensures cart items sync across all devices for the same user
            $query .= " AND c.user_id = :user_id";
            $params['user_id'] = $user_id;
        } elseif ($session_id) {
            // For guest users, check session_id only
            $query .= " AND c.session_id = :session_id";
            $params['session_id'] = $session_id;
        }
        
        $query .= " ORDER BY c.created_at DESC";
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        return $stmt->fetchAll();
    }

    /**
     * Get cart item by product, size, and version
     */
    public function getCartItem($user_id, $session_id, $product_id, $size, $version = null) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE product_id = :product_id AND size = :size";
        
        $params = ['product_id' => $product_id, 'size' => $size];
        
        // Always include version in the query (use empty string if null)
        $query .= " AND version = :version";
        $params['version'] = $version ?: '';
        
        // For logged-in users, only check user_id (not session_id)
        // This ensures cart items sync across all devices for the same user
        if ($user_id) {
            $query .= " AND user_id = :user_id";
            $params['user_id'] = $user_id;
        } elseif ($session_id) {
            // For guest users, check session_id only
            $query .= " AND session_id = :session_id";
            $params['session_id'] = $session_id;
        }
        
        error_log("Cart::getCartItem - Query: $query");
        error_log("Cart::getCartItem - Params: " . json_encode($params));
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        
        if ($result) {
            error_log("Cart::getCartItem - Found existing item: ID {$result['id']}, size: {$result['size']}, version: {$result['version']}");
        } else {
            error_log("Cart::getCartItem - No existing item found");
        }
        
        return $result;
    }

    /**
     * Get cart item by session only (for guest to logged-in user transition)
     */
    private function getSessionCartItem($session_id, $product_id, $size, $version = null) {
        $query = "SELECT * FROM " . $this->table_name . " 
                  WHERE session_id = :session_id AND product_id = :product_id AND size = :size AND version = :version";
        
        $params = [
            'session_id' => $session_id,
            'product_id' => $product_id,
            'size' => $size,
            'version' => $version ?: ''
        ];
        
        error_log("Cart::getSessionCartItem - Query: $query");
        error_log("Cart::getSessionCartItem - Params: " . json_encode($params));
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        
        if ($result) {
            error_log("Cart::getSessionCartItem - Found session item: ID {$result['id']}, user_id: {$result['user_id']}");
        } else {
            error_log("Cart::getSessionCartItem - No session item found");
        }
        
        return $result;
    }

    /**
     * Transfer cart item ownership to a user
     */
    private function transferCartItem($cart_item_id, $user_id, $quantity) {
        $query = "UPDATE " . $this->table_name . " 
                  SET user_id = :user_id, quantity = :quantity 
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $user_id);
        $stmt->bindValue(':quantity', $quantity);
        $stmt->bindValue(':id', $cart_item_id);
        
        if ($stmt->execute()) {
            error_log("Cart::transferCartItem - Successfully transferred item $cart_item_id to user $user_id with quantity $quantity");
            return true;
        } else {
            error_log("Cart::transferCartItem - Failed to transfer item $cart_item_id");
            return false;
        }
    }

    /**
     * Update item quantity
     */
    public function updateQuantity($cart_item_id, $quantity) {
        if ($quantity <= 0) {
            return $this->removeItem($cart_item_id);
        }
        
        $query = "UPDATE " . $this->table_name . " 
                  SET quantity = :quantity 
                  WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':quantity', $quantity, PDO::PARAM_INT);
        $stmt->bindValue(':id', $cart_item_id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Remove item from cart
     */
    public function removeItem($cart_item_id) {
        $query = "DELETE FROM " . $this->table_name . " WHERE id = :id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':id', $cart_item_id, PDO::PARAM_INT);
        
        return $stmt->execute();
    }

    /**
     * Clear cart
     */
    public function clearCart($user_id = null, $session_id = null) {
        $query = "DELETE FROM " . $this->table_name . " WHERE 1=1";
        
        $params = [];
        
        if ($user_id) {
            $query .= " AND user_id = :user_id";
            $params['user_id'] = $user_id;
        } elseif ($session_id) {
            $query .= " AND session_id = :session_id";
            $params['session_id'] = $session_id;
        }
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        return $stmt->execute();
    }

    /**
     * Get cart total with shipping calculation
     */
    public function getCartTotal($user_id = null, $session_id = null, $weight = 0) {
        $query = "SELECT SUM(quantity * price) as subtotal, COUNT(*) as item_count,
                         SUM(quantity) as total_quantity
                  FROM " . $this->table_name . " WHERE 1=1";
        
        $params = [];
        
        if ($user_id) {
            // For logged-in users, only check user_id (not session_id)
            // This ensures cart totals sync across all devices for the same user
            $query .= " AND user_id = :user_id";
            $params['user_id'] = $user_id;
        } elseif ($session_id) {
            // For guest users, check session_id only
            $query .= " AND session_id = :session_id";
            $params['session_id'] = $session_id;
        }
        
        $stmt = $this->conn->prepare($query);
        
        foreach ($params as $key => $value) {
            $stmt->bindValue(':' . $key, $value);
        }
        
        $stmt->execute();
        $result = $stmt->fetch();
        
        $subtotal = $result['subtotal'] ?: 0;
        $item_count = $result['item_count'] ?: 0;
        $total_quantity = $result['total_quantity'] ?: 0;
        
        // Calculate shipping
        $shipping_cost = $this->calculateShippingCost($subtotal, $weight ?: 0.1); // Fixed weight for perfume orders
        
        return [
            'subtotal' => $subtotal,
            'shipping' => $shipping_cost,
            'total' => $subtotal + $shipping_cost,
            'item_count' => $item_count,
            'total_quantity' => $total_quantity
        ];
    }

    /**
     * Calculate shipping cost for cart
     */
    public function calculateShippingCost($orderAmount, $weight = 0) {
        // Get active shipping fees
        $query = "SELECT * FROM shipping_fees 
                  WHERE is_active = 1 
                  AND (min_order_amount <= :order_amount OR min_order_amount = 0)
                  AND (max_order_amount IS NULL OR max_order_amount >= :order_amount OR max_order_amount = 0)
                  ORDER BY sort_order ASC, min_order_amount DESC
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_amount', $orderAmount);
        $stmt->execute();
        $shippingFee = $stmt->fetch();
        
        if (!$shippingFee) {
            return 5.0; // Default shipping fee if no rules found
        }
        
        $feeType = strtolower($shippingFee['fee_type']);
        
        switch ($feeType) {
            case 'weight_based':
                // Use base fee only (no weight calculation)
                return $shippingFee['base_fee'];
            case 'free_shipping_threshold':
                return 0; // Free shipping
            case 'pickup':
                return 0; // Pickup orders are always free
            default:
                return 5.0; // Default fee
        }
    }

    /**
     * Transfer cart from session to user
     */
    public function transferCartToUser($session_id, $user_id) {
        // Update existing cart items to belong to user
        $query = "UPDATE " . $this->table_name . " 
                  SET user_id = :user_id, session_id = NULL 
                  WHERE session_id = :session_id";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':user_id', $user_id, PDO::PARAM_INT);
        $stmt->bindValue(':session_id', $session_id);
        
        return $stmt->execute();
    }

    /**
     * Get cart count
     */
    public function getCartCount($user_id = null, $session_id = null) {
        $result = $this->getCartTotal($user_id, $session_id);
        return $result['item_count'] ?: 0;
    }

    /**
     * Get shipping information for display
     */
    public function getShippingInfo($orderAmount, $weight = 0) {
        // Force fresh query by adding a small random factor to prevent caching
        $query = "SELECT * FROM shipping_fees 
                  WHERE is_active = 1 
                  AND (min_order_amount <= :order_amount OR min_order_amount = 0)
                  AND (max_order_amount IS NULL OR max_order_amount >= :order_amount OR max_order_amount = 0)
                  ORDER BY sort_order ASC, min_order_amount DESC
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_amount', $orderAmount);
        $stmt->execute();
        $shippingFee = $stmt->fetch();
        
        if (!$shippingFee) {
            return [
                'name' => 'Standard Shipping',
                'cost' => 5.0,
                'description' => 'Default shipping rate'
            ];
        }
        
        $cost = $this->calculateShippingCost($orderAmount, $weight);
        
        return [
            'name' => $shippingFee['name'],
            'cost' => $cost,
            'description' => $shippingFee['description'] ?: '',
            'delivery_days_min' => $shippingFee['delivery_days_min'] ?: 1,
            'delivery_days_max' => $shippingFee['delivery_days_max'] ?: 3
        ];
    }

    /**
     * Get shipping configuration for JavaScript
     */
    public function getShippingConfig($orderAmount) {
        $query = "SELECT * FROM shipping_fees 
                  WHERE is_active = 1 
                  AND (min_order_amount <= :order_amount OR min_order_amount = 0)
                  AND (max_order_amount IS NULL OR max_order_amount >= :order_amount OR max_order_amount = 0)
                  ORDER BY sort_order ASC, min_order_amount DESC
                  LIMIT 1";
        
        $stmt = $this->conn->prepare($query);
        $stmt->bindValue(':order_amount', $orderAmount);
        $stmt->execute();
        $shippingFee = $stmt->fetch();
        
        // Debug: Log what shipping fee was found
        error_log("Shipping config query for amount $orderAmount found: " . json_encode($shippingFee));
        
        if (!$shippingFee) {
            return [
                'name' => 'Standard Shipping',
                'base_fee' => 5.0,
                'per_kg_fee' => 0,
                'delivery_days_min' => 2,
                'delivery_days_max' => 3,
                'fee_type' => 'weight_based'
            ];
        }
        
        return [
            'name' => $shippingFee['name'],
            'base_fee' => floatval($shippingFee['base_fee']),
            'per_kg_fee' => floatval($shippingFee['per_kg_fee']),
            'delivery_days_min' => intval($shippingFee['delivery_days_min']),
            'delivery_days_max' => intval($shippingFee['delivery_days_max']),
            'fee_type' => $shippingFee['fee_type']
        ];
    }
}
?>