<?php
require_once __DIR__ . '/../config/payment.php';

class ToyyibPay {
    private $user_secret_key;
    private $api_url;
    private $category_code;
    
    public function __construct() {
        $this->user_secret_key = TOYYIBPAY_USER_SECRET_KEY;
        $this->api_url = TOYYIBPAY_API_URL;
        $this->category_code = TOYYIBPAY_CATEGORY_CODE;
    }
    
    /**
     * Create a category for bills
     */
    public function createCategory($name, $description) {
        $data = [
            'catname' => $name,
            'catdescription' => $description,
            'userSecretKey' => $this->user_secret_key
        ];
        
        $result = $this->makeApiCall('createCategory', $data);
        
        if ($result && isset($result[0]['CategoryCode'])) {
            return $result[0]['CategoryCode'];
        }
        
        return false;
    }
    
    /**
     * Create a bill for payment
     */
    public function createBill($bill_data) {
        $data = [
            'userSecretKey' => $this->user_secret_key,
            'categoryCode' => $this->category_code,
            'billName' => $bill_data['bill_name'],
            'billDescription' => $bill_data['bill_description'],
            'billPriceSetting' => 1, // Fixed amount
            'billPayorInfo' => 1, // Require payer information
            'billAmount' => $bill_data['amount'] * 100, // Convert to cents
            'billReturnUrl' => TOYYIBPAY_RETURN_URL,
            'billCallbackUrl' => TOYYIBPAY_CALLBACK_URL,
            'billExternalReferenceNo' => $bill_data['order_id'],
            'billTo' => $bill_data['customer_name'],
            'billEmail' => $bill_data['customer_email'],
            'billPhone' => $bill_data['customer_phone'],
            'billSplitPayment' => 0,
            'billSplitPaymentArgs' => '',
            'billPaymentChannel' => $this->getPaymentChannel($bill_data['payment_method']),
            'billContentEmail' => 'Thank you for your purchase!',
            'billChargeToCustomer' => 1, // Charge to prepaid
            'billExpiryDays' => TOYYIBPAY_BILL_EXPIRY_DAYS
        ];
        
        // Log the data being sent to API
        error_log("toyyibPay createBill data: " . print_r($data, true));
        
        $result = $this->makeApiCall('createBill', $data);
        
        // Log the API response
        error_log("toyyibPay createBill response: " . print_r($result, true));
        
        if ($result && isset($result[0]['BillCode'])) {
            error_log("toyyibPay createBill success: " . $result[0]['BillCode']);
            
            // Use sandbox URL for sandbox mode
            $payment_domain = (PAYMENT_ENVIRONMENT === 'sandbox') ? 'https://dev.toyyibpay.com/' : 'https://toyyibpay.com/';
            
            return [
                'bill_code' => $result[0]['BillCode'],
                'payment_url' => $payment_domain . $result[0]['BillCode']
            ];
        }
        
        // Check for specific error messages
        if ($result && isset($result[0]['errorMessage'])) {
            $errorMessage = $result[0]['errorMessage'];
            error_log("toyyibPay createBill failed with error: " . $errorMessage);
            
            // Check for common prepaid balance errors
            if (strpos($errorMessage, 'insufficient') !== false || 
                strpos($errorMessage, 'balance') !== false ||
                strpos($errorMessage, 'prepaid') !== false) {
                return [
                    'error' => 'insufficient_prepaid',
                    'message' => 'Insufficient prepaid balance. Please top up your ToyyibPay account.'
                ];
            }
            
            return [
                'error' => 'api_error',
                'message' => $errorMessage
            ];
        }
        
        error_log("toyyibPay createBill failed - no BillCode in response");
        return [
            'error' => 'unknown_error',
            'message' => 'Failed to create payment. Please try again.'
        ];
    }
    
    /**
     * Get bill transactions/status
     */
    public function getBillTransactions($bill_code, $status = null) {
        $data = [
            'billCode' => $bill_code
        ];
        
        if ($status !== null) {
            $data['billpaymentStatus'] = $status;
        }
        
        return $this->makeApiCall('getBillTransactions', $data);
    }
    
    /**
     * Get category details
     */
    public function getCategoryDetails() {
        $data = [
            'userSecretKey' => $this->user_secret_key,
            'categoryCode' => $this->category_code
        ];
        
        return $this->makeApiCall('getCategoryDetails', $data);
    }
    
    /**
     * Set bill to inactive
     */
    public function inactiveBill($bill_code) {
        $data = [
            'secretKey' => $this->user_secret_key,
            'billCode' => $bill_code
        ];
        
        return $this->makeApiCall('inactiveBill', $data);
    }
    
    /**
     * Make API call to toyyibPay
     */
    public function makeApiCall($endpoint, $data) {
        $url = $this->api_url . $endpoint;
        
        // Enhanced logging for debugging
        error_log("toyyibPay makeApiCall - URL: " . $url);
        error_log("toyyibPay makeApiCall - Data: " . print_r($data, true));
        
        $curl = curl_init();
        
        curl_setopt($curl, CURLOPT_POST, 1);
        curl_setopt($curl, CURLOPT_URL, $url);
        curl_setopt($curl, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($curl, CURLOPT_POSTFIELDS, $data);
        curl_setopt($curl, CURLOPT_SSL_VERIFYPEER, false);
        curl_setopt($curl, CURLOPT_TIMEOUT, 30);
        curl_setopt($curl, CURLOPT_USERAGENT, 'PerfumeStore/1.0');
        
        $result = curl_exec($curl);
        $http_code = curl_getinfo($curl, CURLINFO_HTTP_CODE);
        $error = curl_error($curl);
        $info = curl_getinfo($curl);
        
        curl_close($curl);
        
        // Enhanced error logging
        error_log("toyyibPay makeApiCall - HTTP Code: " . $http_code);
        error_log("toyyibPay makeApiCall - Raw Response: " . $result);
        error_log("toyyibPay makeApiCall - cURL Info: " . print_r($info, true));
        
        if ($error) {
            error_log("toyyibPay API cURL Error: " . $error);
            return false;
        }
        
        if ($http_code !== 200) {
            error_log("toyyibPay API HTTP Error: " . $http_code . " - Response: " . $result);
            return false;
        }
        
        $decoded = json_decode($result, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            error_log("toyyibPay API JSON Error: " . json_last_error_msg() . " - Raw: " . $result);
            return false;
        }
        
        error_log("toyyibPay makeApiCall - Decoded Response: " . print_r($decoded, true));
        return $decoded;
    }
    
    /**
     * Get payment channel based on method
     */
    private function getPaymentChannel($payment_method) {
        switch ($payment_method) {
            case 'toyyibpay_fpx':
                return '0'; // FPX only
            case 'toyyibpay_card':
                return '1'; // Credit Card only
            case 'toyyibpay_both':
            default:
                return '2'; // Both FPX and Credit Card
        }
    }
    
    /**
     * Set category code
     */
    public function setCategoryCode($category_code) {
        $this->category_code = $category_code;
    }
    
    /**
     * Get category code
     */
    public function getCategoryCode() {
        return $this->category_code;
    }
    
    /**
     * Validate payment status
     */
    public function validatePaymentStatus($status) {
        $valid_statuses = ['1', '2', '3', '4'];
        return in_array($status, $valid_statuses);
    }
    
    /**
     * Get status description
     */
    public function getStatusDescription($status) {
        $statuses = [
            '1' => 'Successful',
            '2' => 'Pending',
            '3' => 'Failed',
            '4' => 'Pending'
        ];
        
        return $statuses[$status] ?? 'Unknown';
    }
    
    /**
     * Test API connectivity (for debugging)
     */
    public function testApiConnection() {
        $test_data = [
            'userSecretKey' => $this->user_secret_key,
            'billCode' => 'test123' // This will fail but we'll see the response
        ];
        
        return $this->makeApiCall('getBillTransactions', $test_data);
    }
    
    /**
     * Test category access (for debugging)
     */
    public function testCategoryAccess() {
        $bill_data = [
            'userSecretKey' => $this->user_secret_key,
            'categoryCode' => $this->category_code
        ];
        
        return $this->makeApiCall('getBills', $bill_data);
    }
}
?>