<?php
require_once 'models/Product.php';
require_once 'config/database.php';

// Get all products from database
$product = new Product();
$allProducts = $product->getAllProducts();

// Get categories from database
$conn = getDB();
// Check if sort_order column exists, if not use id for ordering
$checkColumnQuery = "SHOW COLUMNS FROM categories LIKE 'sort_order'";
$checkStmt = $conn->query($checkColumnQuery);
$sortOrderExists = $checkStmt->rowCount() > 0;

if ($sortOrderExists) {
    $categoriesQuery = "SELECT * FROM categories WHERE is_active = 1 ORDER BY sort_order ASC, name ASC";
} else {
    $categoriesQuery = "SELECT * FROM categories WHERE is_active = 1 ORDER BY id ASC, name ASC";
}

$categoriesStmt = $conn->query($categoriesQuery);
$categories = $categoriesStmt->fetchAll();

// Debug: Check if categories are being fetched
if (empty($categories)) {
    // Try without is_active filter in case the column doesn't exist
    $fallbackQuery = "SELECT * FROM categories ORDER BY id ASC, name ASC";
    $fallbackStmt = $conn->query($fallbackQuery);
    $categories = $fallbackStmt->fetchAll();
}

// Fallback sample data if database is empty
if (empty($allProducts)) {
    $allProducts = [
        [
            'id' => 1,
            'name' => 'Angel Spirit',
            'description' => 'A luxurious floral fragrance with notes of vanilla and oriental spices',
            'price' => '89.00',
            'image' => 'images/angel-spirit-300x300.png',
            'volume' => '30ml',
            'category' => 'Floral',
            'brand' => '2 Days Parfume',
            'rating' => 4.5,
            'review_count' => 25,
            'is_bestseller' => 1,
            'is_new_arrival' => 0
        ],
        [
            'id' => 2,
            'name' => 'Suave',
            'description' => 'A fresh and woody fragrance perfect for everyday wear',
            'price' => '69.00',
            'image' => 'images/suave.png',
            'volume' => '35ml',
            'category' => 'Woody',
            'brand' => '2 Days Parfume',
            'rating' => 4.2,
            'review_count' => 18,
            'is_bestseller' => 1,
            'is_new_arrival' => 0
        ],
        [
            'id' => 3,
            'name' => 'BVLGARI Rose Goldea',
            'description' => 'An elegant rose fragrance with oriental undertones',
            'price' => '229.00',
            'image' => 'images/placeholder.svg',
            'volume' => '50ml',
            'category' => 'Floral',
            'brand' => 'BVLGARI',
            'rating' => 4.8,
            'review_count' => 42,
            'is_bestseller' => 0,
            'is_new_arrival' => 1
        ],
        [
            'id' => 4,
            'name' => 'Chopard Happy Bigaradia',
            'description' => 'A fresh citrus fragrance with sparkling notes',
            'price' => '249.00',
            'image' => 'images/placeholder.svg',
            'volume' => '50ml',
            'category' => 'Citrus',
            'brand' => 'Chopard',
            'rating' => 4.6,
            'review_count' => 31,
            'is_bestseller' => 0,
            'is_new_arrival' => 1
        ],
        [
            'id' => 5,
            'name' => 'Chanel No.5',
            'description' => 'The iconic aldehydic floral fragrance',
            'price' => '399.00',
            'image' => 'images/placeholder.svg',
            'volume' => '100ml',
            'category' => 'Floral',
            'brand' => 'Chanel',
            'rating' => 4.9,
            'review_count' => 156,
            'is_bestseller' => 1,
            'is_new_arrival' => 0
        ],
        [
            'id' => 6,
            'name' => 'Dior Sauvage',
            'description' => 'A fresh and spicy fragrance for the modern man',
            'price' => '299.00',
            'image' => 'images/placeholder.svg',
            'volume' => '60ml',
            'category' => 'Fresh',
            'brand' => 'Dior',
            'rating' => 4.7,
            'review_count' => 89,
            'is_bestseller' => 1,
            'is_new_arrival' => 0
        ],
        [
            'id' => 7,
            'name' => 'Tom Ford Black Orchid',
            'description' => 'A dark and mysterious oriental fragrance',
            'price' => '349.00',
            'image' => 'images/placeholder.svg',
            'volume' => '50ml',
            'category' => 'Oriental',
            'brand' => 'Tom Ford',
            'rating' => 4.4,
            'review_count' => 67,
            'is_bestseller' => 0,
            'is_new_arrival' => 0
        ],
        [
            'id' => 8,
            'name' => 'Versace Eros',
            'description' => 'A fresh and spicy fragrance with woody notes',
            'price' => '279.00',
            'image' => 'images/placeholder.svg',
            'volume' => '100ml',
            'category' => 'Spicy',
            'brand' => 'Versace',
            'rating' => 4.3,
            'review_count' => 73,
            'is_bestseller' => 0,
            'is_new_arrival' => 0
        ]
    ];
}

// Get filter parameters
$category = isset($_GET['category']) ? $_GET['category'] : '';
$search = isset($_GET['search']) ? $_GET['search'] : '';
$sort = isset($_GET['sort']) ? $_GET['sort'] : 'name';

// Build filters for Product model
$filters = [];

// Handle special categories (bestseller, new)
if ($category === 'bestseller') {
    $filters['bestsellers'] = true;
} elseif ($category === 'new') {
    $filters['new_arrivals'] = true;
} elseif (!empty($category)) {
    // Regular category filter using slug
    $filters['category'] = $category;
}

if (!empty($search)) {
    $filters['search'] = $search;
}

// Get filtered products from database using Product model
if (!empty($filters)) {
    $filteredProducts = $product->getAll($filters);
} else {
    $filteredProducts = $allProducts;
}

// Sort products
switch ($sort) {
    case 'price_low':
        usort($filteredProducts, function($a, $b) {
            // Use display_price if available, otherwise min_price, otherwise 0
            $priceA = floatval($a['display_price'] ?? $a['min_price'] ?? 0);
            $priceB = floatval($b['display_price'] ?? $b['min_price'] ?? 0);
            return $priceA <=> $priceB;
        });
        break;
    case 'price_high':
        usort($filteredProducts, function($a, $b) {
            // Use display_price if available, otherwise max_price, otherwise 0
            $priceA = floatval($a['display_price'] ?? $a['max_price'] ?? 0);
            $priceB = floatval($b['display_price'] ?? $b['max_price'] ?? 0);
            return $priceB <=> $priceA;
        });
        break;
    case 'name':
    default:
        usort($filteredProducts, function($a, $b) {
            return strcmp($a['name'], $b['name']);
        });
        break;
}

$totalProducts = count($filteredProducts);

// Debug information (remove in production)
$debugInfo = [
    'total_products' => count($allProducts),
    'filtered_products' => count($filteredProducts),
    'category_filter' => $category,
    'search_filter' => $search,
    'sort_filter' => $sort
];
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Product Catalog - Perfume Store</title>
    <link rel="stylesheet" href="styles.css">
    <link rel="stylesheet" href="products.css">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0/css/all.min.css" rel="stylesheet">
    <script src="script.js" defer></script>
</head>
<body class="product-catalog-page">
    <!-- Top Navigation -->
    <nav class="top-nav">
        <div class="nav-left">
            <i class="fas fa-bars" id="hamburger-menu"></i>
        </div>
        <div class="nav-center">
            <div class="logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="logo-img">
            </div>
        </div>
        <div class="nav-right">
            <div class="search-cart-container">
                <i class="fas fa-shopping-cart cart-icon" onclick="window.location.href='cart.php'">
                    <span class="cart-count" id="cart-count">0</span>
                </i>
            </div>
        </div>
    </nav>

    <!-- Mobile Menu Overlay -->
    <div class="mobile-menu-overlay" id="mobile-menu-overlay">
        <div class="mobile-menu" id="mobile-menu">
        <div class="mobile-menu-header">
            <div class="mobile-logo">
                <img src="images/LOGO.png" alt="2 Days Parfume Logo" class="mobile-logo-img">
            </div>
            <button class="mobile-menu-close" id="mobile-menu-close">
                <i class="fas fa-times"></i>
            </button>
        </div>
        <div class="mobile-menu-content">
            <ul class="mobile-menu-list">
                <li><a href="index.php" class="mobile-menu-item"><i class="fas fa-home"></i> Home</a></li>
                <li><a href="bestseller.php" class="mobile-menu-item"><i class="fas fa-star"></i> Best Sellers</a></li>
                <li><a href="product-catalog.php" class="mobile-menu-item"><i class="fas fa-box"></i> All Products</a></li>
                <li><a href="favorites.php" class="mobile-menu-item"><i class="fas fa-heart"></i> Favorites</a></li>
                <li><a href="cart.php" class="mobile-menu-item"><i class="fas fa-shopping-cart"></i> Cart</a></li>
                <li><a href="login.php" class="mobile-menu-item"><i class="fas fa-sign-in-alt"></i> Login</a></li>
                <li><a href="signup.php" class="mobile-menu-item"><i class="fas fa-user-plus"></i> Sign Up</a></li>
                <li><a href="#" class="mobile-menu-item"><i class="fas fa-cog"></i> Settings</a></li>
            </ul>
        </div>
        </div>
    </div>

    <!-- Main Content -->
    <main class="main-content">
        <!-- Page Header -->
        <div class="page-header">
            <h1>Product Catalog</h1>
            <p class="page-subtitle">Discover our complete collection of premium fragrances</p>
        </div>

        <!-- Search and Filter Section -->
        <section class="search-section">
            <div class="search-container">
                <div class="search-input-group">
                    <input type="text" id="search-input" placeholder="Search products..." class="search-input" value="<?php echo htmlspecialchars($search); ?>">
                    <button class="filter-btn" id="filter-btn" onclick="toggleFilterDropdown()">
                        <i class="fas fa-filter"></i>
                    </button>
                    <button class="search-btn" onclick="performSearch()">
                        <i class="fas fa-search"></i>
                    </button>
                </div>
                
                <!-- Filter Dropdown -->
                <div class="filter-dropdown" id="filter-dropdown">
                    <div class="filter-row">
                        <label for="category-filter">Category:</label>
                        <select id="category-filter" class="filter-select" onchange="performSearch()">
                            <option value="">All Categories</option>
                            <?php foreach ($categories as $cat): ?>
                                <option value="<?php echo htmlspecialchars($cat['slug']); ?>" 
                                        <?php echo $category === $cat['slug'] ? 'selected' : ''; ?>>
                                    <?php echo htmlspecialchars($cat['name']); ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="filter-row">
                        <label for="sort-filter">Sort by:</label>
                        <select id="sort-filter" class="filter-select" onchange="performSearch()">
                            <option value="name" <?php echo $sort === 'name' ? 'selected' : ''; ?>>Sort by Name</option>
                            <option value="price_low" <?php echo $sort === 'price_low' ? 'selected' : ''; ?>>Price: Low to High</option>
                            <option value="price_high" <?php echo $sort === 'price_high' ? 'selected' : ''; ?>>Price: High to Low</option>
                        </select>
                    </div>
                </div>
            </div>
        </section>

        <!-- Results Header -->
        <div class="results-header">
            <div class="results-info">
                <span class="results-count"><?php echo $totalProducts; ?> products found</span>
            </div>
            <div class="view-toggle">
                <button class="view-btn active" id="grid-view-btn" onclick="switchView('grid')" title="Grid View">
                    <i class="fas fa-th"></i>
                </button>
                <button class="view-btn" id="list-view-btn" onclick="switchView('list')" title="List View">
                    <i class="fas fa-list"></i>
                </button>
            </div>
        </div>

        <!-- Debug Information (remove in production) -->
        <!-- <div style="background: #f0f0f0; padding: 8px; margin: 5px 20px; border-radius: 5px; font-size: 12px;">
            <strong>Debug Info:</strong><br>
            Total Products: <?php echo $debugInfo['total_products']; ?><br>
            Filtered Products: <?php echo $debugInfo['filtered_products']; ?><br>
            Category Filter: <?php echo $debugInfo['category_filter'] ?: 'None'; ?><br>
            Search Filter: <?php echo $debugInfo['search_filter'] ?: 'None'; ?><br>
            Sort Filter: <?php echo $debugInfo['sort_filter']; ?>
        </div> -->

        <!-- Products Grid -->
        <section class="products-section">
            <?php if (empty($filteredProducts)): ?>
                <div class="no-products">
                    <i class="fas fa-search"></i>
                    <h3>No products found</h3>
                    <p>Try adjusting your search or filter criteria</p>
                </div>
            <?php else: ?>
                <div class="products-grid" id="products-grid">
                    <?php foreach ($filteredProducts as $index => $product): ?>
                    <?php include 'includes/product-card.php'; ?>
                    <?php endforeach; ?>
                </div>
                <div class="products-list" id="products-list" style="display: none;">
                    <?php foreach ($filteredProducts as $index => $product): ?>
                    <?php include 'includes/product-list-item.php'; ?>
                    <?php endforeach; ?>
                </div>
            <?php endif; ?>
        </section>
    </main>

    <!-- Bottom Navigation -->
    <nav class="bottom-nav">
        <div class="nav-item" onclick="window.location.href='index.php'">
            <i class="fas fa-home"></i>
        </div>
        <div class="nav-item active">
            <i class="fas fa-box"></i>
        </div>
        <div class="nav-item" onclick="window.location.href='cart.php'">
            <i class="fas fa-shopping-cart"></i>
        </div>
        <div class="nav-item" onclick="checkRegisteredUserAndRedirect('royalty-rewards.php')">
            <i class="fas fa-crown"></i>
        </div>
    </nav>

    <script>
        function performSearch() {
            const search = document.getElementById('search-input').value;
            const category = document.getElementById('category-filter').value;
            const sort = document.getElementById('sort-filter').value;
            
            const params = new URLSearchParams();
            if (search) params.append('search', search);
            if (category) params.append('category', category);
            if (sort) params.append('sort', sort);
            
            window.location.href = 'product-catalog.php?' + params.toString();
        }

        function toggleFilterDropdown() {
            const dropdown = document.getElementById('filter-dropdown');
            const filterBtn = document.getElementById('filter-btn');
            
            if (dropdown.style.display === 'block') {
                dropdown.style.display = 'none';
                filterBtn.classList.remove('active');
            } else {
                dropdown.style.display = 'block';
                filterBtn.classList.add('active');
            }
        }

        // Close dropdown when clicking outside
        document.addEventListener('click', function(event) {
            const dropdown = document.getElementById('filter-dropdown');
            const filterBtn = document.getElementById('filter-btn');
            
            if (!dropdown.contains(event.target) && !filterBtn.contains(event.target)) {
                dropdown.style.display = 'none';
                filterBtn.classList.remove('active');
            }
        });

        // Enter key support for search
        document.getElementById('search-input').addEventListener('keypress', function(e) {
            if (e.key === 'Enter') {
                performSearch();
            }
        });

        function viewProduct(productId) {
            window.location.href = 'product-detail.php?id=' + productId;
        }

        function switchView(viewType) {
            const gridView = document.getElementById('products-grid');
            const listView = document.getElementById('products-list');
            const gridBtn = document.getElementById('grid-view-btn');
            const listBtn = document.getElementById('list-view-btn');
            
            if (viewType === 'grid') {
                gridView.style.display = 'grid';
                listView.style.display = 'none';
                gridBtn.classList.add('active');
                listBtn.classList.remove('active');
            } else {
                gridView.style.display = 'none';
                listView.style.display = 'block';
                gridBtn.classList.remove('active');
                listBtn.classList.add('active');
            }
        }
    </script>

    <!-- Analytics Tracking Script -->
    <script src="/js/analytics-tracking.js"></script>
</body>
</html>